/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon.util;

//standard Java stuff
import java.io.*;
import java.util.*;

import talon.*;
import talon.components.*;
import talon.components.exceptions.*;
import talon.implementations.*;

/**
 * Base class for handling properties. 
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
 */
public abstract class BasePropertyManager extends BaseComponent implements Component, PropertyManager  {

    private Hashtable props = new Hashtable();

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Object getProperty( String name ) {
        return this.props.get( name );
    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Object getProperty( String name, Object _default ) {

        Object value = (String)this.props.get( name );

        if ( value == null ) {
            value = _default;
        } 

        return value;
    }
    
    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public boolean getBoolean( String name ) {
        String value = getString( name );

        //if it is null assume false.
        if ( value == null ) {
            return false;
        } 
        
        return value.equalsIgnoreCase( "true" );
        
    }


    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setBoolean( String name, boolean value ) {

        if ( value ) {
            setProperty( name, "true" );            
        } else {
            setProperty( name, "false" );
        }
        return this;
    }


    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setString( String name, String value ) {
        setProperty( name, value );
        return this;
    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public String getString( String name ) {
        return (String)getProperty( name );
    }


    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public String getString( String _name, String _default ) {

        String value = getString( _name );

        if ( value == null ) {
            return _default;
        } else {
            return value;
        }
        
    }
    
    
    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Vector getVector( String name ) {

        Vector v = new Vector();

        int i = 0;
        while( true ) {

            String prop = getString( name + "." + i );

            //terminate the while loop if we have no more variables.
            if ( prop == null) {
                break;
            } 

            v.addElement( prop );
            
            ++i;
        }

        return v;
    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager getGroup( String key ) {

        Enumeration keys = getKeys();
        SimplePropertyManager result = new SimplePropertyManager();

        boolean noKeysFound = true;
        
        while ( keys.hasMoreElements() ) {

            String name = (String)keys.nextElement();

            //test if the current name is valid.
            if ( name.indexOf( key ) == 0 ) {

                noKeysFound = false;
                
                Object value = getProperty( name );

                //update the name.
                name = name.substring( key.length() + 1, name.length() );

                result.setProperty( name, value );
            } 
            
        } 
        
        //if no values were found... return null
        if ( noKeysFound ) {
            return null;
        } else {
            return result;             
        }


    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setProperty( String name, Object value ) {

        if ( name == null || value == null ) {
            throw new IllegalArgumentException( "Name and value must not be null" );
        } 

        this.props.put( name, value );
        
        return this;
    }
    
    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Enumeration getKeys() {
        return this.props.keys();
    }
    
    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public void setProperties( Properties props ) {
        this.props = props;
    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public void require( String[] requirements ) throws RequiredPropertyException {

        for ( int i = 0; i < requirements.length; ++i ) {
            if ( ! this.props.containsKey( requirements[i] ) ) {
                throw new RequiredPropertyException( "The required property for this component does not exist: " + requirements[i] );
            } 
            
        } 

    }

    /**
     * @see PropertyManager
     * @see Hashtable
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public boolean containsKey( String key ) { 
        return props.containsKey( key );
    }

    /**
     * @see PropertyManager
     * @see Hashtable
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public boolean contains( Object object ) { 
        return props.containsKey( object );
    }

    
    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public void dump() {
        dump( System.out );
    }

    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public void dump( PrintStream ps) {

        ps.println( getPrettyPrint() );
    }
    
    /**
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    private String getPrettyPrint() {

        Enumeration keys = getKeys();

        StringBuffer buff = new StringBuffer();

        buff.append( "-- listing properties --\n" );
        
        while ( keys.hasMoreElements() ) {

            String name = (String)keys.nextElement();
            
            Object value = getProperty( name );

            buff.append( name.toString() + " = " + value.toString() + "\n" );
            
        } 

        return buff.toString();
    }


    /**
     * @see PropertyManager
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public int size() {
        return this.props.size();
    }

    
    /**
     * get a Integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Integer getInteger( String _name ) {

        return (Integer)getProperty( _name );
        
    }

    /**
     * get a string property but specify a default.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Integer getInteger( String _name, Integer _default ) {

        Integer result = getInteger( _name );

        if ( result == null ) {
            result = _default;
        } 
        
        return result;
    }
    
    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setInteger( String _name, Integer _value ) {

        this.setProperty( _name, _value );

        return this;
    }

    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setInteger( String _name, int _value ) {

        return setInteger( _name, new Integer( _value ) );

    }


    /**
     * get a Long property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Long getLong( String _name ) {

        return (Long)getProperty( _name );
        
    }

    /**
     * get a string property but specify a default.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public Long getLong( String _name, Long _default ) {

        Long result = getLong( _name );

        if ( result == null ) {
            result = _default;
        } 
        
        return result;
    }
    
    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setLong( String _name, Long _value ) {

        this.setProperty( _name, _value );

        return this;
    }

    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BasePropertyManager.java,v 1.12 2001/04/14 18:03:50 burton Exp $
     */
    public PropertyManager setLong( String _name, long _value ) {

        return setLong( _name, new Long( _value ) );

    }


    
}
