/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon.implementations;

import java.io.*;
import java.util.*;

import talon.*;
import talon.components.*;


/**
 * Simple logging abstraction.  There are more powerful logging mechanisms
 * (Turbine, Log4J, Avalon, etc) but this is very simple and stable.
 *
 * <p>
 * This object support the following initialization properties:
 *
 *     - file     -> The file to use for output.  if this can't be use the
 *                   Component will throw an Exception on startup.
 *     - debug    -> If true.  Enable debug output.
 *     - stdout   -> If true, log to stdout.
 *     - stderr   -> If true, log to stderr
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
 */
public class SimpleLogger extends BaseComponent implements Component, Logger {

    public static final String[] REQUIRED_PROPERTIES = { "project" };
    
    public static final String INIT_PROPERTY_FILE = "file";

    public static final String INIT_PROPERTY_DEBUG = "debug";
    public static final String INIT_PROPERTY_STDERR = "stdout";
    public static final String INIT_PROPERTY_STDOUT = "stdout";


    public static final String INIT_PROPERTY_PROJECT = "project";
    
    /**
     * Set this to false if we have any problems with output to a file.
     */
    private boolean fileSafe = true;

    /**
     * The file we need to log to.
     */
    private String file = null;
    
    /**
     * Stores the PrintWriter used when logging to file.
     */
    private PrintWriter fileOutput = null;

    /**
     * True if we are in debug mode.
     */
    private boolean debug = false;

    private boolean stdout = false;

    private boolean stderr = false;

    private String project = null;


    /**
     * Default constructor when used as a Component
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public SimpleLogger() { }

    /**
     * Default constructor when used as an Object
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public SimpleLogger( ComponentHandle handle,
                         boolean debug,
                         boolean stdout,
                         boolean stderr,
                         String project ) {

        this.setComponentHandle( handle );
        this.debug = debug;
        this.stdout = stdout;
        this.stderr = stderr;
        this.project = project;
    }
    
    /**
     * Perform any necessary initialization
     *
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void init() throws TalonException {

        getComponentHandle().getInitProperties().require( REQUIRED_PROPERTIES );
        
        this.debug = this.getComponentHandle()
            .getInitProperties().getBoolean( INIT_PROPERTY_DEBUG );

        this.stdout = this.getComponentHandle()
            .getInitProperties().getBoolean( INIT_PROPERTY_STDOUT );

        this.stderr = this.getComponentHandle()
            .getInitProperties().getBoolean( INIT_PROPERTY_STDERR );

        this.file = this.getComponentHandle()
            .getInitProperties().getString( INIT_PROPERTY_FILE );

        this.project = this.getComponentHandle().getInitProperties()
            .getString( INIT_PROPERTY_PROJECT );

    }

        
    /**
     * Log a message
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void message( String message ) {
        message( null, message );
    }
        
    /**
     * Log a message
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void message( Object source, String message ) {
        logString( source, getLogPrefix() + " -> " + message );
    }
    
    /**
     * Log an error
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void error( String message ) {
        error( null, message );
    }

    /**
     * Log an error
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void error( Object source, String message ) {
        logString( source, getLogPrefix() + " -> ERROR: " + message );
    }

    
    /**
     * Log a warning
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void warning( String message ) {
        warning( null, message );
    }

    /**
     * Log a warning
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void warning( Object source, String message ) {
        logString( source, getLogPrefix() + " -> WARNING: " + message );
    }
        
    /**
     * Log a debug message
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void debug( String message ) {
        debug( null, message );
    }

    /**
     * Log a debug message
     *
     * @author <A HREF="mailto:burton@apache.org">Kevin A. Burton</A>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void debug( Object source, String message ) {

        if ( debug ) {
            logString( source, getLogPrefix() + " -> DEBUG: " + message );
        } 

    }
    
    /**
     * @see #logString
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    private void logString( String s ) {
        logString( null, s);
    }
    
    /**
     * Log the specified string to all necessary channels.
     *
     * @param source Object source where this was logged from
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    private void logString( Object source, String s ) {

        if ( source != null ) {
            s = source.getClass().getName() + " : " + s;
        } 
        
        if ( stdout ) {
            System.out.println( s );
        } 
        
        if ( stderr ) {
            System.err.println( s );
        } 

        //determine if we should log to file.
        if ( isFileEnabled() ) {

            // FIXME: this will break when we migrate to a property mechanism
            // that can reload on-the-fly.  if the user changes the file that
            // will be written to Logger won't realize this and will continually
            // write to the old file.  Write a unit test for making sure the
            // file can change on the fly.
            if ( fileOutput == null ) {

                try {

                    fileOutput = new PrintWriter( new FileWriter( this.file, true ) );

                    this.debug( "Logging to disk is enabled: " + this.file );
                    
                } catch ( Throwable t ) {

                    //mark file logging as unsafe and then
                    this.fileSafe = false;
                    this.error( t.getMessage() );
                    
                }
                
            }

            fileOutput.println( s );
            fileOutput.flush(); //flush everytime because we don't know when the VM will shut down..
            
        } 
        
    }
    
    /**
     * Get the prefix for log output.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    private String getLogPrefix() {
        return this.project + " [" + this.getDate() + "]";
    }

    
    /**
     *
     * Determine if it is OK to log to disk
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    private boolean isFileEnabled() {

        //if it isn't safe... return right away
        if ( fileSafe == false ) {
            return false;
        } 
        

        if ( this.file == null ) {
            return false;
        } 
        
        File file = new File( this.file );

        boolean exists = file.exists();

        boolean canWrite = file.canWrite();

        //determine if it is safe for logging
        fileSafe = exists && canWrite;
        
        //perform debug checks...
        if ( exists == false ) {

            this.error( "Log file does not exist: " + this.file );

        } else if ( canWrite == false ) {

            this.error( "Log file can not be written to: " + this.file );

        } 
        
        return fileSafe;
        
    }


    /**
     * Get the date/time in a nice format for logging.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    private String getDate() {

        //format should be DAY-MONTH-YEAR-HOUR-MIN

        Calendar cal = Calendar.getInstance();
        StringBuffer date = new StringBuffer( "" );

        date.append( cal.get( Calendar.MONTH ) + 1 );        
        date.append( "-" );
        date.append( cal.get( Calendar.DAY_OF_MONTH ) );
        date.append( "-" );
        date.append( cal.get( Calendar.YEAR ) );
        date.append( "-" );
        date.append( cal.get( Calendar.HOUR ) + 1);
        date.append( "." );

        int min = cal.get( Calendar.MINUTE );

        //if the minute is less than ten it should be 01, 02, etc.  always two digits
        if ( min < 10 ) {
            date.append( 0 );
        } 

        date.append( min );

        date.append( "." );
        date.append( cal.get( Calendar.SECOND ) );

        date.append( "." );
        date.append( cal.get( Calendar.MILLISECOND ) );
        
        return date.toString();
            
    }


    
    /**
     * @see Logger
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SimpleLogger.java,v 1.8 2001/04/14 18:03:49 burton Exp $
     */
    public void error( Throwable t ) {

        ByteArrayOutputStream bos = new ByteArrayOutputStream();

        t.printStackTrace( new PrintStream( bos ) );


        //FIXME: this needs to actually pretty print the message line by line.
        //This works for now though.  
        this.message( bos.toString() );
        
    }

    
}
