/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon.implementations;

//standard Java stuff
import java.io.*;
import java.util.*;

import talon.*;
import talon.components.*;
import talon.util.*;


/**
 * Used when working with property files.
 *
 * <p>
 * This object depends on the following initialization properties:
 *
 * - project -> Name of the project this property file is goign to be used in.
 * This is used so that we can find a [PROJECT].properties file in user.home,
 * java.home or /etc.
 *
 * <p> Generally this object should be deployed as a Singleton... maybe a Named
 * Singleton
 * 
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
 */
public class FilePropertyManager extends BasePropertyManager {

    public static final String[] REQUIRED_PROPERTIES = { "project" };
    
    public static final String INIT_PROPERTY_PROJECT = "project";

    private String propertyFilename = null;
    
    /**
     * Constructor when used as a Component
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
     */
    public FilePropertyManager( ) { } 
    
    /**
     * Constructor when used as an Object
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
     */
    public FilePropertyManager( ComponentHandle handle,
                                String project ) throws TalonException {
        
        this.setComponentHandle( handle );
        handle.getInitProperties().setProperty( "project", project );
        this.init();
    } 

    /**
     * Perform any necessary initialization
     *
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
     */
    public void init() throws TalonException {

        getComponentHandle().getInitProperties().require( REQUIRED_PROPERTIES );
        
        String project = this.getComponentHandle().getInitProperties().getString( INIT_PROPERTY_PROJECT );

        String[] locations = getFileLocations( project );
        
        String file = null;
        
        for( int i = 0; i < locations.length; ++i ) {

            if ( new File( locations[i] ).exists() ) {
                file = locations[i];
            } 
                        
        }

        //if the .properties file can not be found... throw an Exception
        if ( file == null ) {
            throw new TalonException( "Unable to find a property file to use." );
        } 
        
        //keep track of the file this was initialized from.
        this.propertyFilename = file;
        
        try {

            Properties props = new Properties();
            props.load( new FileInputStream( file ) );

            this.setProperties( props );
            
        } catch ( Exception e ) {
            e.printStackTrace();
            throw new TalonException( e.getMessage() );
        }
           
    }

    /**
     * Given a project name, determine where its files might be stored.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
     */
    private String[] getFileLocations( String project ) {

        
        Vector v = new Vector();

        //if there is a System Property defined named PROJECT_PROPERTIES... use that

        String prop = project.toUpperCase() + "_PROPERTIES";
        String value = System.getProperty( prop );
        
        if ( value != null ) {
            v.addElement( value );
        } 
        
        v.addElement( System.getProperty( "user.home" ) +
                      File.separator +
                      project +
                      ".properties" );

        v.addElement( System.getProperty( "java.home" ) +
                      File.separator +
                      project +
                      ".properties" );

        v.addElement( "/etc/" +
                      project +
                      ".properties" );


        String[] s = new String[v.size()];
        v.copyInto(s);

        return s;
        
    }

    /**
     * Get the property file that this was initialized from. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: FilePropertyManager.java,v 1.5 2001/04/14 18:03:49 burton Exp $
     */
    public String getPropertyFilename() {
        return this.propertyFilename;
    }
    
}
