/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon.components;

import java.io.*;
import java.util.*;
import talon.*;
import talon.interfaces.*;
import talon.components.exceptions.*;

/**
 * Class similar to java.lang.Properties
 *
 @ @see Hashtable
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
 */
public interface PropertyManager extends Component, Dumpable {

    /**
     * Get a given property from parsed props.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Object getProperty( String name );

    /**
     * Get a given property from parsed props.  Return default if it is not defined.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Object getProperty( String name, Object value );

    /**
     * Set a property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setProperty( String name, Object value );

    /**
     * Get a property but assume it is a boolean
     * 
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public boolean getBoolean( String name );

    
    /**
     * Set a boolean property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setBoolean( String namme, boolean value );

    /**
     * get a string property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public String getString( String _name );

    /**
     * get a string property but specify a default.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public String getString( String _name, String _default );
    
    /**
     * Set a string property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setString( String _name, String _value );

    
    /**
     * get a Integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Integer getInteger( String _name );

    /**
     * get a string property but specify a default.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Integer getInteger( String _name, Integer _default );
    
    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setInteger( String _name, Integer _value );

    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setInteger( String _name, int _value );


    /**
     * get a Long property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Long getLong( String _name );

    /**
     * get a string property but specify a default.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Long getLong( String _name, Long _default );
    
    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setLong( String _name, Long _value );

    /**
     * Set a integer property
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager setLong( String _name, long _value );

    
    /**
     * <p>
     * Get a property but assume it is multivalued.  A good example is a set or
     * properties such as:
     *
     * <p>
     * <pre>
     * example.0=foo
     * example.1=bar
     * </pre>
     *
     * <p>
     * This would return a Vector which contained "foo" and "bar" if you
     * requested a name of "example".
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Vector getVector( String name );

    /**
     * <p>
     * Similar to getVector but returns every property with the given key. 
     *
     * <p>
     * <pre>
     * example.0.foo_name=foo_value
     * example.0.bar_name=bar_value
     * </pre>
     *
     * <p>
     * This would return a Hashtable which contained "foo_name" and "bar_var" if you
     * requested a name of "example.0".  Note that the key is removed so that
     * this Hashtable can be passed in a logical manner.
     *
     * <p>
     * If no values are found then this should return null.
     *
     * @see getVector#
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public PropertyManager getGroup( String key );

    /**
     * Get all known keys for registered properties.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public Enumeration getKeys();

    /**
     * Require that the given properties exist within this PropertyManager.  if
     * They are not found then we should throw and Exception.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public void require( String[] properties ) throws RequiredPropertyException;

    /**
     * Return true if the given key exists within this PropertyManager
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public boolean containsKey( String key );

    /**
     * Return true if the given object exists within this PropertyManager
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public boolean contains( Object object );


    /**
     * Return the number of object held.
     *
     * @see Hashtable
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: PropertyManager.java,v 1.2 2001/04/14 18:03:49 burton Exp $
     */
    public int size();
    
}
