/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon;

import talon.components.*;
import talon.implementations.*;

import java.util.*;

/**
 * Handles intializing Talon and performing all startup procedures. 
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: Initializer.java,v 1.9 2001/04/14 18:03:48 burton Exp $
 */
public class Initializer {


    public final static String TALON_LOG_DEBUG_KEY = "talon.log.debug";

    public static final String TALON_LOG_STDOUT_KEY = "talon.log.stdout";

    public static final String TALON_LOG_STDERR_KEY = "talon.log.stderr";

    public static final String TALON_PROJECT_KEY = "talon.project";
    
    private static boolean initialized = false;

    /**
       Keep track of talon being initialized and don't do this twice.
     */
    private static boolean working = false;
    

    /**
     * Initialize Talon if necessary.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Initializer.java,v 1.9 2001/04/14 18:03:48 burton Exp $
     */
    public static void init() {

        if ( shouldInitialize() ) {

            //fetch default components from the factory so they are at least instantiated.

            Initializer.working = true;
            
            try {

                //get the system property for the project or "talon" if none specified.
                String project = System.getProperty( TALON_PROJECT_KEY );
                if ( project == null ) {
                    project = "talon";
                } 

                FilePropertyManager pm = new FilePropertyManager( ComponentFactory.TALON_DEFAULT_PROPERTY_MANAGER,
                                                                  project );

                ComponentFactory.putSingleton( ComponentFactory.TALON_DEFAULT_PROPERTY_MANAGER, (Component)pm );
                
                Logger logger = new SimpleLogger( ComponentFactory.TALON_DEFAULT_LOGGER,
                                                  pm.getBoolean( TALON_LOG_DEBUG_KEY ),
                                                  pm.getBoolean( TALON_LOG_STDOUT_KEY ),
                                                  pm.getBoolean( TALON_LOG_STDERR_KEY ),
                                                  pm.getString( TALON_PROJECT_KEY ) );

                ComponentFactory.setLogger( logger );
                
                ComponentFactory.putSingleton( ComponentFactory.TALON_DEFAULT_LOGGER, (Component)logger );
                
                logger.message( "Talon initialized via file " + pm.getPropertyFilename() );

                logger.message( pm.toString() );
                logger.message( logger.toString() );

                //loop through all possible components... adding handles for them.

                logger.message( "Creating handles for all deployed components..." );

                int i = 0;

                while ( true ) {

                    String group_name = "talon.component." + i;
                    
                    PropertyManager group = pm.getGroup( group_name );
                    
                    if ( group == null ) {
                        break;
                    } 

                    logger.debug( "Building properties for: " + group_name );
                    
                    //create the component handle

                    ComponentHandle handle = new ComponentHandle();
                    handle.setName( group.getString( ComponentHandle.DEPLOY_PROPERTY_NAME ));
                    handle.setInterface( group.getString( ComponentHandle.DEPLOY_PROPERTY_INTERFACE ) );
                    handle.setImplementation( group.getString( ComponentHandle.DEPLOY_PROPERTY_IMPLEMENTATION ) );
                    handle.setLifetime( group.getString( ComponentHandle.DEPLOY_PROPERTY_LIFETIME ) );
                    handle.setDescription( group.getString( ComponentHandle.DEPLOY_PROPERTY_DESCRIPTION ) );

                    //add the rest of the parameters to the component...

                    Enumeration keys = group.getKeys();
                    while ( keys.hasMoreElements() ) {
                        String name = (String)keys.nextElement();
                        String value = group.getString( name );

                        logger.debug( "\t" + name + " = " + value );

                        handle.getInitProperties().setProperty( name, value );
                    } 
                    
                    logger.message( handle.toString() );

                    HandleManager.registerComponentHandle( handle );
                    
                    ++i;
                } 
                
                
                
            } catch ( Throwable t ) {
                t.printStackTrace();
            }

        }         

    }

    /**
     * Return true if Talon is initialized.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Initializer.java,v 1.9 2001/04/14 18:03:48 burton Exp $
     */
    public static boolean isInitialized() {
        return initialized;
    }

    /**
     * Return true if Talon is working.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Initializer.java,v 1.9 2001/04/14 18:03:48 burton Exp $
     */
    public static boolean isWorking() {
        return working;
    }

    /**
     * Return true if Talon should be initialized.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Initializer.java,v 1.9 2001/04/14 18:03:48 burton Exp $
     */
    public static boolean shouldInitialize() {
        return isWorking() == false && isInitialized() == false;
    }
    
}
