/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon;

import java.io.*;
import java.net.*;
import java.util.*;
import talon.implementations.SimplePropertyManager;
import talon.components.*;
import talon.util.net.*;

/**
 * Responsible for keeping track of handles and providing methods for fetching
 * them. 
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
 */
public class HandleManager extends BaseComponent  {

    private static SimplePropertyManager byName = new SimplePropertyManager();
    private static SimplePropertyManager byInterface = new SimplePropertyManager();
    private static SimplePropertyManager byImplementation = new SimplePropertyManager();


    //required for Dumpable interface
    public void init() throws TalonException { }
    
    public static void registerComponentHandle( ComponentHandle handle ) throws TalonException {

        Logger logger = ComponentFactory.getLogger();
        
        logger.message( "registering handle:" + handle.toString() );
        
        //register by name...
        if ( handle.getName() != null ) {
            logger.message( "registering handle (byName):" + handle.toString() );

            registerComponentHandle( byName, handle.getName(), handle );
        } 

        //register by interface
        if ( handle.getInterface() != null ) {
            logger.message( "registering handle (byInterface):" + handle.toString() );
            registerComponentHandle( byInterface, handle.getInterface(), handle );
        } 

        //register by implementation
        if ( handle.getImplementation() != null ) {
            logger.message( "registering handle (byImplementation):" + handle.toString() );
            registerComponentHandle( byImplementation, handle.getImplementation(), handle );
        } 
        
    }

    /**
     * If the given handles does NOT already exist.  Add it.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    private static void registerComponentHandle( SimplePropertyManager manager,
                                                 String key,
                                                 ComponentHandle handle )
        throws TalonException {

        if ( manager.contains( key ) ) {
            throw new TalonException( "HandleManager already contains this component: " + handle.toString() );
        } else {
            manager.setProperty( key, handle );
        }
        
    }
    
    /**
     * Get a handle by its name
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    public static ComponentHandle byName( String _name )
        throws TalonException {
        
        ComponentHandle handle = (ComponentHandle)byName.getProperty( _name );

        if ( handle == null ) {
            throw new NotFoundException( "ComponentHandle not found: " + _name );
        } 
        
        
        return handle;

    }

    /**
     * Get a handle by its interface
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    public static ComponentHandle byInterface( String _interface )
        throws TalonException { 

        return (ComponentHandle)byInterface.getProperty( _interface );

    }

    /**
     * Get a handle by its impl
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    public static ComponentHandle byImplementation( String _implementation )
        throws TalonException {

        ComponentHandle handle = (ComponentHandle)byImplementation.getProperty( _implementation );

        if ( handle == null ) {
            throw new NotFoundException( "ComponentHandle not found: " + _implementation );
        } 
        
        return handle;

    }

    /**
     * @see #byImplementation( String _implementation )
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    public static ComponentHandle byImplementation( Class _implementation )
        throws TalonException {

        return byImplementation( _implementation.getName() );
        
    }

    
    /**
     * Get a specic registry.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    static PropertyManager getNameRegistry() { 
        return byName;
    }

    /**
     * Get a specic registry.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    static PropertyManager getInterfaceRegistry() { 
        return byInterface;
    }


    /**
     * Get a specic registry.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    static PropertyManager getImplementationRegistry() { 
        return byImplementation;
    }
    
    /**
     * @see Dumpable
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
    public static void dump() {
        dump(System.out);
    }


    /**
     * @see Dumpable
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: HandleManager.java,v 1.10 2001/04/14 18:03:48 burton Exp $
     */
     public static void dump( PrintStream ps ) {

         ps.println( " --- BEGIN Dumping HandleManager --- ");
         
         //dump byName
         dump( ps, byName, "\n\n\n\tBY NAME\n\n\n");
         //dump byInterface
         dump( ps, byInterface, "\n\n\n\tBY INTERFACE\n\n\n");

         //dump byImplementation
         dump( ps, byImplementation, "\n\n\n\tBY IMPLEMENTATION\n\n\n");
         
         ps.println( " --- END Dumping HandleManager --- ");
     }

    private static void dump( PrintStream ps, SimplePropertyManager spm, String name ) {

        ps.println( name );

        Enumeration keys = spm.getKeys();

        while ( keys.hasMoreElements() ) {

            Object key = keys.nextElement();

            Object value = spm.getProperty( key.toString() );

            ps.print( "\t\t" );
            ps.println( key.toString() );

            ps.print( "\t\t" );
            ps.println( value.toString() );

            ps.println();
            
            
            ((ComponentHandle)value).getInitProperties().dump( ps );
            
        } 
        
    }

    
}
