/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon;


import java.io.*;
import java.net.*;
import java.util.*;
import talon.references.*;
import talon.implementations.*;
import talon.components.*;
import talon.util.net.*;

/**
 * Object which is used to find components within the system.  Each Component
 * has a ComponentHandle which is used within the system to create instances and
 * to hold property information.          
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
 */
public class ComponentHandle {
    
    public static final String DEPLOY_PROPERTY_INTERFACE = "talon_interface";
    public static final String DEPLOY_PROPERTY_IMPLEMENTATION = "talon_implementation";
    public static final String DEPLOY_PROPERTY_LIFETIME = "talon_lifetime";
    public static final String DEPLOY_PROPERTY_NAME = "talon_name";
    public static final String DEPLOY_PROPERTY_DESCRIPTION = "talon_description";


    public static final String LIFETIME_DEMAND = "demand";

    public static final String LIFETIME_SINGLETON = "singleton";
    
    private String _interface = null;
    
    private String _implementation = null;

    private String _lifetime = null;

    private String _description = null;

    private URI typeReference = null;
    
    private PropertyManager initProperties = new SimplePropertyManager();
    
    /**
     * Support for named components.  Handles can have names that are each
     * different from each other.  This means you could have say multiple
     * singletons each point to an different XML document, or to a different
     * URI.  Default is none.
     * 
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    private String name = null;

    /**
     * Create a basic ComponentHandle
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle() {}
     
    /**
     * ComponentHandle with just an interface.  The classname to instantiate is
     * pulled from the Talon properties.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle( String _interface ) {

        this._interface = _interface;

    }

    /**
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle( String _interface,
                            String _implementation ) {

        this( _interface );
        this._implementation = _implementation;

    }


    /**
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle( String _interface,
                            String _classname,
                            PropertyManager initProperties ) {

        this( _interface, _classname );
        this.initProperties = initProperties;
    }

    /**
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle( String name,
                            String _interface,
                            String _classname,
                            PropertyManager initProperties ) {

        this( _interface, _classname, initProperties );
        this.setName( name );
    }

    
    
    /**
     * Get the Interface of this classname.  Note that in some cases this may
     * not be set because the interface name is not required to instantiate a
     * component. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public String getInterface() {
        return _interface;
    }
    
    /**
     * Set the value of _interface.
     * @param v  Value to assign to _interface.
     */
    public ComponentHandle setInterface( String _interface ) {
        this._interface = _interface;
        return this;
    }

    /**
     * Get the value of _classname.
     * @return value of _classname.
     */
    public String getImplementation() {
        return this._implementation;
    }
    
    /**
     * Set the value of _classname.
     * @param v  Value to assign to _classname.
     */
    public ComponentHandle setImplementation( String _implementation ) {
        this._implementation = _implementation;
        return this;
    }

    /**
     * Set the value of _classname.
     * @param v  Value to assign to _classname.
     */
    public ComponentHandle setImplementation( Class _implementation ) {
        this.setImplementation( _implementation.getName() );
        return this;
    }
    
    /**
     * Get the name for this component. 
     */
    public String getName() {
        return this.name;
    }

    /**
     * Set the name for this component. 
     */
    public ComponentHandle setName( String name ) {
        this.name = name;
        return this;
    }

     /**
      * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
      * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
      */
    public String getLifetime() { 
        return this._lifetime;
    }

    

    /**
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle setLifetime( String _lifetime ) { 
        this._lifetime = _lifetime;
        return this;
    }

    /**
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public String getDescription() {
        return this._description; 
    }
    
    /**
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle setDescription( String _description ) {
        this._description = _description; 
        return this;
    }

    /**
     * Maintains a set of properties that can be passed to a component during
     * initialization.  
     * 
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public PropertyManager getInitProperties() {
        return this.initProperties;
    }

    /**
     * @see #getInitProperties
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle setInitProperties( PropertyManager initProperties ) {
        this.initProperties = initProperties;
        return this;
    }
    
    /**
     * Dump Handle information.. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public String toString() {
        return this.getTypeReference().toString();
    }

    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public URI getTypeReference() {

        if ( this.typeReference == null ) {
            typeReference = TypeReferenceFactory.getTypeReference( this );
        } 

        return this.typeReference;
    }
    
    
    /**
     * Return a cloned ComponentHandle. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentHandle.java,v 1.15 2001/04/14 18:03:48 burton Exp $
     */
    public Object clone() {

        ComponentHandle handle = new ComponentHandle();

        handle.setName( this.getName() );
        handle.setDescription( this.getDescription() );
        handle.setLifetime( this.getLifetime() );
        handle.setInitProperties( this.getInitProperties() );
        handle.setInterface( this.getInterface() );
        handle.setImplementation( this.getImplementation() );
        
        return (Object)handle;
        
    }
    
}
