/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon;

import java.io.*;
import java.net.*;
import java.util.*;
import talon.components.*;
import talon.implementations.*;
import talon.interfaces.*;
import talon.util.net.*;

/**
 * Create instances of components and manage their lifetimes.
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
 */
public class ComponentFactory {

    public static final String LIFETIME_SINGLETON = "singleton";
    public static final String LIFETIME_DEMAND = "demand";
    
    // **************************************
    // BEGIN DEFAULT TALON COMPONENTS
    // **************************************

    public static final ComponentHandle TALON_DEFAULT_LOGGER = 
        new ComponentHandle().setName( "TALON_DEFAULT_LOGGER" )
                             .setInterface( "talon.components.Logger" )
                             .setImplementation( "talon.implementations.SimpleLogger" )
                             .setInitProperties( new SimplePropertyManager().setProperty( "project", "talon" ) )
                             .setLifetime( LIFETIME_SINGLETON );

    
    public static final ComponentHandle TALON_DEFAULT_PROPERTY_MANAGER = 
        new ComponentHandle().setName( "TALON_DEFAULT_PROPERTY_MANAGER" )
                             .setInterface( "talon.components.PropertyManager" )
                             .setImplementation( "talon.implementations.SimplePropertyManager" )
                             .setInitProperties( new SimplePropertyManager().setProperty( "project", "talon" ) )
                             .setLifetime( LIFETIME_SINGLETON );

    // **************************************
    // END DEFAULT TALON COMPONENTS
    // **************************************

    /**
       Logger implementation.
     */
    public static final ComponentHandle LOGGER_INTERFACE_KEY =
        new ComponentHandle( "talon.components.Logger" );

    /**
       PropertyManager implementation.
     */
    public static final ComponentHandle PROPERTY_MANAGER_INTERFACE_KEY =
        new ComponentHandle( "talon.components.PropertyManager" );


    private static Hashtable singletons = new Hashtable();

    private static Logger logger = null;
    
    /**
     * Get a Component via class.  This will create the component on demand.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    public static Component getInstance( Class _class )
        throws TalonException {

        return getInstance( _class.getName() );
        
    }

    
    /**
     * Get a Component via classname.  This will create the component on demand.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    public static Component getInstance( String classname )
        throws TalonException {

        return getInstance( new ComponentHandle().setImplementation( classname )
                                                 .setLifetime( LIFETIME_DEMAND ) );
        
    }
    
    /**
     * Given a ComponentHandle get a Component
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    public static Component getInstance( ComponentHandle handle )
        throws TalonException {


        //init Talon... this only executes once (if talon isn't already initialized)
        Initializer.init();
        
        //make sure this handle has an implementation.  if the implementation is
        //not set then we should try to use the HandleManager to recover.


        //use the default lifetime
        if ( handle.getLifetime() == null ) {
            handle.setLifetime( LIFETIME_DEMAND );
        }

        //Try to fill in any blank information.  A Component can be instantiated
        //by multiple methods. (name, interface, implementation) and we need  to
        //try to fill in any blank information.
        

        constrain( handle.getName(), handle, HandleManager.getNameRegistry() );
        constrain( handle.getImplementation(), handle, HandleManager.getImplementationRegistry() );


        //no go over the handle and make sure we have enough information.
        
        
        //if the implementatio is unknown there is nothing we can do
        if ( handle.getImplementation() == null ) {
            throw new TalonException( "Unable to determine implementation classname." );
        } 
        
        //determine what lifetime to use

        String lifetime = handle.getLifetime();
        
        if ( lifetime.equals( LIFETIME_DEMAND ) ) {
            return getDemandBasedInstance( handle );
        } else if ( lifetime.equals( LIFETIME_SINGLETON ) ) {
            return getSingletonBasedInstance( handle );
        } 

        throw new TalonException( "Do not know how to create " + handle.toString() );
    }

    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    private static void constrain( String name,
                                   ComponentHandle handle, 
                                   PropertyManager repository ) {

        if ( name != null && repository.containsKey( name ) ) {

            
            ComponentHandle prototype = (ComponentHandle)repository.getProperty( name );

            constrain( handle, prototype);                
            
        } 

        
    }
    

    /**
     * Assume that the given handle does not have all known information.  Use
     * the given prototype as a basis for getting more info.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    private static void constrain( ComponentHandle handle,
                                   ComponentHandle prototype ) {

        //name
        if ( prototype.getName() != null && handle.getName() == null ) {
            handle.setName( prototype.getName() );
        } 
        
        //interface
        if ( prototype.getInterface() != null && handle.getInterface() == null ) {
            handle.setInterface( prototype.getInterface() );
        } 
                
        //implementation
        if ( prototype.getImplementation() != null && handle.getImplementation() == null ) {
            handle.setImplementation( prototype.getImplementation() );
        } 

        //consider merging init properties sometime in the future.

        if ( handle.getInitProperties().size() == 0 && prototype.getInitProperties().size() != 0 ) {

            logger.debug( "Handle provided doesn't have any properties." );
            handle.setInitProperties( prototype.getInitProperties() );
                    
        } 

        
    }
    
    /**
     * Get a component instantiating a new one everytime we need it.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    private static Component getDemandBasedInstance( ComponentHandle handle )
        throws TalonException { 

        //since this component is always created... just return it.
        return instantiateComponent( handle );
        
    }

    
    /**
     * Get a singleton or create it if necessary. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    private static Component getSingletonBasedInstance( ComponentHandle handle )
        throws TalonException { 
        
        Component comp = (Component)singletons.get( handle );
        
        if ( comp == null ) {
            //we have to create it!

            comp = instantiateComponent( handle );
            
            putSingleton( handle, comp );
        } 
            
        return comp;
            
    }

    /**
     * Perform basic instantiation on a component. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    private static Component instantiateComponent( ComponentHandle handle )
        throws TalonException {

        try {

            Component comp = (Component)Class.forName( handle.getImplementation() ).newInstance();
            comp.setComponentHandle( handle );
            comp.init();

            return comp;
        } catch ( Throwable t ) {
            throw new TalonException( t );
        }
        
    }
    
    /**
     * Put a singleton to the Factory.
     * FIXME: Should this throw an Exception if this object is already present?
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    static void putSingleton( ComponentHandle handle, Component comp ) {
        singletons.put( handle, comp );
    }

    /**
     * A generic component with Talon has access to the default logger stream.
     * Some internal Talon classes ( not components ) might want to get this
     * stream so it can log.  This is provided for easy of use so that they
     * don't have to worry about fetching an object from the component system.  
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    public static Logger getLogger() {
        return ComponentFactory.logger;
    }
    
    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ComponentFactory.java,v 1.17 2001/04/14 18:03:48 burton Exp $
     */
    public static void setLogger( Logger logger ) {
        ComponentFactory.logger = logger;
    }
    
}
