/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */


package talon;

import java.util.*;

import talon.components.*;
import talon.util.net.*;

/**
 * Any object that wants to act as a Component with Talon must implement this
 * interface.  This contains all necessary methods for an object to participate
 * as a first class Component with the system.
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
 */
public interface Component {

    /**
     * Called when this object is initialized.  Generally a Component should
     * perform no operations within a constructor (any constructor).  In fact
     * Talon relies on the fact that a constructor has not been called.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public void init() throws TalonException;

    /**
     * Get the handle for this component.  
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle getComponentHandle();

    /**
     * <p>
     * Get a UNIQUE reference for this object.  The general goal for this
     * method is that within a given connected component system (even a P2P
     * system) each object can be refered to in a unique manner.
     *
     * <p> The reference may be used to get refer to an object with a runtime
     * system.  Depending on its lifetime (demand, singleton, etc) it may have
     * been garbage collected so you can not be sure that it will still be
     * around.
     *
     * <p>
     * Should return a URI like talon://HOST/component/CLASSNAME/interface/INTERFACE/hashcode/HASHCODE
     *
     * @see Object#hashCode
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public URI getComponentReference();

    /**
     * <p>
     * Similer to getComponentReference but does not include any instance
     * specific information.  This is necessary if you want to request a object
     * by its type information.
     * 
     * <p>
     * Should return a URI like talon:///component/CLASSNAME/interface/INTERFACE
     * 
     * @see #getComponentReference
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public URI getTypeReference();
    
    /**
     * Set the handle for this component.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public void setComponentHandle( ComponentHandle handle );

    /**
     * Get the Talon default log stream for this component.  Any logging your
     * component has to do should use this method.  This allows Components to
     * switch logging implementations without needing a specific configuration. 
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    public Logger getLogger();

    /**
     * Get the list of Component interfaces that this component implements.
     * This should is done progmatically and only return 'Classes' that are
     * components.
     * 
     * @see Class#getInterfaces
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: Component.java,v 1.11 2001/04/14 18:03:48 burton Exp $
     */
    //public Class[] getInterfaces();

}

