/*
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package talon;

import java.util.*;
import java.net.*;

import talon.references.*;
import talon.components.*;
import talon.implementations.*;
import talon.util.net.*;

/**
 * Implements most of the Component interface. Should be used by component
 * develoeprs as a base for the Components they create.
 *
 * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
 * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
 */
public abstract class BaseComponent implements Component {


    private ComponentHandle handle  = null;

    private URI componentReference = null;

    private URI typeReference = null;


    /**
     * The known Component interfaces that this Component implements.
     */
    private Class[] componentInterfaces = null;


    /**
     * Don't provide an implementation by default.
     * 
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public void init() throws TalonException { }
    
    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public ComponentHandle getComponentHandle() {
        return this.handle;
    }
    
    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public void setComponentHandle( ComponentHandle handle ) {
        this.handle = handle;
    }

    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public URI getComponentReference() {

        //determine this URI if it isn't set.
        if ( componentReference == null ) {
            componentReference = ComponentReferenceFactory.getComponentReference( this );
        } 

        return componentReference;
        
    }


    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public URI getTypeReference() {

        if ( this.typeReference == null ) {
            typeReference = TypeReferenceFactory.getTypeReference( this );
        } 

        return this.typeReference;
    }
    
    /**
     * @see Component
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public Logger getLogger() {
        try {

            return (Logger)ComponentFactory
                .getInstance( ComponentFactory.TALON_DEFAULT_LOGGER );

        } catch ( TalonException t ) {

            //if this throws an Exception we have big problems caused by Talon
            // not being initialized.  It isn't amazingly important that we
            // handle this because Talon will likely terminate itself. 
            t.printStackTrace();
            return null;

        }
    }
    
    /**
     * Default impl of toString() just dumps handle info...
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: BaseComponent.java,v 1.13 2001/04/14 18:03:48 burton Exp $
     */
    public String toString() {
        return this.getComponentHandle().toString();
    }

    
}

