/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */


package org.openprivacy.reptile.xml;

import java.io.*;
import java.util.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import org.openprivacy.panther.components.*;

import talon.*;
import talon.util.*;
import talon.util.net.*;

import org.openprivacy.reptile.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

/**
 * Responsible for fetching content form the cache.
 * 
 * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
 * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
 */
public class SequenceDispatchHandler extends DefaultHandler {

    public static final URI RESOURCE = new URI( "resource:/xml/sequence-dispatch.xml" );

    private String requestedSequenceName = null;
    
    private Hashtable sequenceDispatchEntries = new Hashtable();

    private ParameterPasser params = null;
    
    //event level variables...

    private String currentSequenceName = "";

    /**
     * 
     * @param requestedSequenceName The value of the the sequence we want
     * dispatched.
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    public SequenceDispatchHandler( String requestedSequenceName,
                                    ParameterPasser params ) throws ReptileException {

        this.requestedSequenceName = requestedSequenceName;
        
        this.params = params;

        try {

            //FIXME: don't parse every time...
            
            Proxy proxy = ReptileResources.getInstance().getProxy();
            
            XMLReader xmlr = ReptileParserFactory.getXMLReader();

            xmlr.setContentHandler( this );

            InputSource is = new InputSource( proxy.get( RESOURCE ).getInputStream() );

            xmlr.parse( is );

        } catch ( Throwable t ) {

            Log.error( t );
            
            throw new ReptileException( t );
            
        } 
        
    }
    
    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    public void startElement( String uri,
                              String local,
                              String raw,
                              Attributes attrs ) {

        //only operate on the correct namespace.
        if ( uri.equals( ReptileResources.SEQUENCE_DISPATCH_NAMESPACE ) ) {

            //find sequences...
            if ( local.equals( "sequence" ) ) {

                String name = attrs.getValue( "name" );
                String parent = attrs.getValue( "parent" );

                String cacheable = attrs.getValue( "cacheable" );

                this.currentSequenceName = name;

                SequenceDispatchEntry sde = new SequenceDispatchEntry( name );
                sde.setParent( parent );

                if ( cacheable != null ) {

                    sde.setCacheable( Boolean.getBoolean( cacheable ) );
                    
                } 

                this.setSequenceDispatchEntry( currentSequenceName, sde );
                
            } 

            //process skeletons...
            if ( local.equals( "skeleton" ) ) {

                this.getSequenceDispatchEntry( currentSequenceName )
                    .setSkeleton( attrs.getValue( "location" ) );

            } 

            //process stylesheets.
            if ( local.equals( "stylesheet" ) ) {

                String stylesheet = attrs.getValue( "location" );
                
                this.getSequenceDispatchEntry( currentSequenceName )
                    .getStylesheetStack().addStylesheet( stylesheet );

            } 

            //process params
            if ( local.equals( "parameter" ) &&
                 requestedSequenceName.equals( currentSequenceName ) ) {

                String pname = attrs.getValue( "name" );

                String pvalue = attrs.getValue( "value" );

                if ( pname != null && pvalue != null ) {

                    params.put( pname, pvalue );
                    
                } 
                
            } 
            
        } 
        
    } 
    
    /**
     * Dispatch this sequence after it is found...
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    public String dispatch() throws Exception {
        
        //make sure that we actually have the sequence that is necessary.

        SequenceDispatchEntry root = getSequenceDispatchEntry( requestedSequenceName );

        if ( root == null ) {

            throw new Exception( "The necessary sequence is not available: " + requestedSequenceName );
            
        } 

        //now to through the order vector and construct a new, normalized SequenceDispatchEntry.

        SequenceDispatchEntry real = normalize( root );        

        return StylesheetEngineFactory.getStylesheetEngine().transform( real,
                                                                        this.params );
    }

    /**
     * Normalize the given SequenceDispatchEntry resolving all parent entries
     * (if any).
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    private SequenceDispatchEntry normalize( SequenceDispatchEntry sde ) throws Exception {

        //FIXME: when we implement DOM caching we should support caching the
        //normalization of these...

        Log.debug( "normaling SequenceDispatchEntry" );
        
        SequenceDispatchEntry result = new SequenceDispatchEntry( sde.getName() );

        normalize( sde, result );
        
        //now compute the order
        SequenceDispatchEntry current = sde;

        Log.debug( "BEGIN normalizing..." );

        String parent = null;
        
        while ( true ) {

            //update the skeleton
            if ( current.getSkeleton() != null )
                result.setSkeleton( current.getSkeleton() );
            
            normalize( current, result );
            
            parent = current.getParent();

            Log.debug( "Normalizing parent: " + parent );
            
            if ( parent == null ) 
                break;
            
            current = getSequenceDispatchEntry( parent );
            
            if ( current == null )
                throw new Exception( "The necessary sequence is not available: " + parent );
            
        }

        Log.debug( "END normalizing..." );
        
        return result;        

    }

    /**
     * Migrate any stylesheets in the original into the result.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    private SequenceDispatchEntry normalize( SequenceDispatchEntry original,
                                               SequenceDispatchEntry result ) {

        //get its StylesheetStack and go through its entries.
        StylesheetStack ss = original.getStylesheetStack();

        Iterator si = ss.getIterator();
        
        while ( si.hasNext() ) {

            result.getStylesheetStack().addStylesheet( (URI)si.next() );
                
        }

        return result;
        
    }
        
    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    public SequenceDispatchEntry getSequenceDispatchEntry( String name ) {

        SequenceDispatchEntry entry = (SequenceDispatchEntry)this.sequenceDispatchEntries.get( name );

        if ( entry == null )
            Log.warn( "Could not find sequence: " + name );        
        
        return entry;

    }

    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceDispatchHandler.java,v 1.17 2001/12/12 09:41:21 burton Exp $
     */
    public void setSequenceDispatchEntry( String name, SequenceDispatchEntry entry  ) {

        this.sequenceDispatchEntries.put( name, entry );

    }
    
}
