/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.xml;

import java.io.*;
import java.util.*;

import org.xml.sax.*;
import org.xml.sax.helpers.*;

import talon.*;
import talon.util.*;
import talon.util.net.*;

import org.openprivacy.reptile.*;
import org.openprivacy.reptile.util.*;

import javax.xml.transform.*;
import javax.xml.transform.sax.*;
import javax.xml.transform.stream.*;

import org.openprivacy.panther.components.*;

/**
 * Responsible for creating TrAX Transformers with all the features necessary
 * within Reptile.
 * 
 * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
 * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
 */
public class ReptileTransformerFactory {
    
    /**
     * Holds all entries for ProxyEntries.
     */
    private Hashtable proxyEntryCache = new Hashtable();

    private Hashtable templatesCache = new Hashtable();

    private Proxy proxy = null;
    
    /**
     * Instance member for <code>ReptileTransformerFactory</code>
     */
    private static ReptileTransformerFactory instance = null;

    private TransformerFactory transformerFactory = null;
    
    /**
     * 
     * Create a new <code>ReptileTransformerFactory</code> instance.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    public ReptileTransformerFactory() {
        
        this.proxy = ReptileResources.getInstance().getProxy();
        
    }

    /**
     * @see Transformer
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    public Transformer getTransformer( URI stylesheet ) throws ReptileException {

        try {

            Transformer transformer = this.getTemplates( stylesheet.toString() ).newTransformer();

            //we need to use panther as a URIResolver

            transformer.setURIResolver( proxy.getURIResolver() );
            
            return transformer;

        } catch ( Exception e ) {

            Log.error( e );
            
            throw new ReptileException( e );                    
            
        } 
        
    }

    /**
     * @see TransformerHandler
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    public TransformerHandler getTransformerHandler( String stylesheet ) throws Exception {

        SAXTransformerFactory stf = (SAXTransformerFactory)transformerFactory;

        //FIXME: I think I fould the problem!!! Could the problem be that we are
        //not resolving the stylesheet with a correct resolver 
        
        TransformerHandler th = stf.newTransformerHandler( getTemplates( stylesheet ) );

        th.getTransformer().setURIResolver( proxy.getURIResolver() );

        return th;
        
    }
    
    /**
     * Get a template, caching and reloading when necessary.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    private Templates getTemplates( String stylesheet ) throws Exception {

        Log.debug( "Getting template for: " + stylesheet, this );

        //
        ProxyEntry pe = (ProxyEntry)proxyEntryCache.get( stylesheet );

        if ( pe == null ) {

            pe = proxy.get( stylesheet );
            proxyEntryCache.put( stylesheet, pe );
            
        } 
        
        if ( pe.hasChanged() ) {

            Log.message( "Template changed, updating cache: " + stylesheet, this );
            
            //update the ProxyEntry we have cached.
            proxyEntryCache.put( stylesheet, proxy.get( stylesheet ) );

            return newTemplates( stylesheet );
            
        } else {

            Templates templates = (Templates)templatesCache.get( stylesheet );

            if ( templates == null ) {

                return newTemplates( stylesheet);
                
            } else {

                Log.debug( "Found templates in cache: " + stylesheet, this );

                return templates;
                
            }
            
        }
        
    }

    /**
     * Use a Transformer to create a Templates.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    private Templates newTemplates( String stylesheet ) throws Exception {

        XMLReader xmlReader = ReptileParserFactory.getXMLReader();

        InputSource is = new InputSource( proxy.get( stylesheet ).getInputStream() );
        
        Source source = new SAXSource( xmlReader, is  );
        
        Templates templates = getTransformerFactory().newTemplates( source );

        //update the templatesCache with the new template
        templatesCache.put( stylesheet, templates );
        
        return templates;
        
    }

    /**
     * Create a new TransformerFactory...
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    public TransformerFactory getTransformerFactory() {

        if ( transformerFactory == null ) {

            //we need to synchronize around the current singleton.  We wouldn't
            //want two thread to hit this point while the system is creating its
            //transformerFactory
            synchronized( this ) {
            
                //don't constantly create a new Transformer Factory.  since this
                //is a singleton we should be able to reuse this.
                
                this.transformerFactory = TransformerFactory.newInstance();

                this.transformerFactory.setURIResolver( proxy.getURIResolver() );

            }
            
        } 

        return this.transformerFactory;

    }

    /**
     * Get the current TransformerFactory implementation classname we are using.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String getTransformerFactoryImplementation() {

        return getTransformerFactory().getClass().getName();

    }
    
    /**
     * Flush the templates cache.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public void flushTemplatesCache() {

        this.templatesCache = new Hashtable();
        
    }

    /**
     * Return the size of the templatesCache
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getTemplatesCacheSize() {

        return this.templatesCache.size();
        
    }
    
    /**
     * 
     * Get an instance of the <code>ReptileTransformerFactory</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ReptileTransformerFactory.java,v 1.11 2002/03/09 23:24:40 burton Exp $
     */
    public static ReptileTransformerFactory getInstance() {
        
        if ( instance == null )
            instance = new ReptileTransformerFactory();
        
        return instance;
        
    }
    
}
