/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.tasks;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.panther.components.*;

import talon.*;
import talon.components.*;
import talon.util.*;
import talon.util.net.*;

import org.openprivacy.reptile.util.*;

/**
 * Handles actually running the necessary tasks as threads.
 *
 * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
 * @version $Id: TaskThread.java,v 1.23 2002/07/06 05:57:57 burton Exp $
 */
public class TaskThread extends Thread {

    /**
     * Sleep for 10 seconds when Reptile first starts up so that full
     * initialization can take place.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public static final long INITIAL_SLEEP = 10 * 1000;

    /**
     * Length of time to sleep between tasks runs.  Most of the time we will
     * just continue to sleep.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public static final long SLEEP = 30 * 1000; // 30 seconds
    
    private static ThreadGroup group = new ThreadGroup( "urn:reptile/threadgroup/tasks" );
    
    /**
     * Create a new <code>TaskThread</code> instance
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public TaskThread() {

        super( group, "TaskThread" );

        Log.message( "TaskThread created...", this );

        //we need to make threads daemon threads so that the VM can shutdown
        //correctly.
        super.setDaemon( true );
        
    }

    /**
     * Run the thread and continually execute the task.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     */
    public void run() {

        try { 

            initialsleep();
            
            while ( true ) {

                if ( TaskManager.checkTaskSystemDisabled() )
                    return;

                //run all tasks...
                this.runTasks();

                synchronized( this ) {

                    this.wait( SLEEP );
                
                }

            } 
            
        } catch ( Throwable t ) {

            Log.debug( t.getMessage(), this );
            
        }
        
    }

    /**
     * Run all necessary tasks (those registered in TaskManager).
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void runTasks() {

        Enumeration tasks = TaskManager.getInstance().getTasks();

        while ( tasks.hasMoreElements() ) {

            //return if the tasks system has been shutdown.
            if ( TaskManager.checkTaskSystemDisabled() )
                return;
            
            Task task = (Task)tasks.nextElement();

            synchronized( task ) {
            
                TaskManager.runTask( task );
                
            }
            
        }
        
    }

    /**
     * Display all tasks. 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void displaytasks() {

        TaskManager tm = TaskManager.getInstance();
        
        Log.message( "TaskThread: the following " + tm.getTaskCount() + " tasks are known within the TaskManager: ", this );

        Enumeration tasks = tm.getTasks();
        
        while ( tasks.hasMoreElements() ) {

            Task task = (Task)tasks.nextElement();

            Log.message( "TaskThread:    " + task.getIdentifier(), this );
            
        }
        
    }
    
    /**
     * Sleep for INITIAL_SLEEP milliseconds.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void initialsleep() {

        Log.message( "TaskThread: Sleeping ... " + INITIAL_SLEEP, this );

        try { 

            synchronized( this ) {
                
                this.wait( INITIAL_SLEEP );
                
            }

        } catch ( Throwable t ) {
            
            t.printStackTrace();
            
        }

    }
    
}
