/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.tasks;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.panther.components.*;

import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.om.*;

import talon.*;
import talon.components.*;
import talon.util.*;
import talon.util.net.*;

import org.apache.log4j.Category;

/**
 * Generic interface for running tasks in the background.  These are basically
 * just Threads.  
 *
 * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
 * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
 */
public class TaskManager {

    /**
     * Instance member for <code>TaskManager</code>
     */
    private static TaskManager instance = null;

    private Hashtable tasks = new Hashtable();
    
    /**
     * 
     * Get an instance of the <code>TaskManager</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
     */
    public static TaskManager getInstance() {
        
        if ( instance == null ) {
            
            instance = new TaskManager();
            
        }
        
        return instance;
        
    }

    /**
     * Return true if this Task manager contains the given task.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
     */
    public boolean contains( String identifier ) {

        return this.tasks.containsKey( identifier );

    }

    /**
     * Get all known tasks.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public Enumeration getTasks() {

        return this.tasks.elements();

    }
    
    /**
     * Start a specific task.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
     */
    public void register( Class clazz ) throws Exception {

        register( (Task)clazz.newInstance() );
        
    }

    /**
     * Register the given task for further processing.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
     */
    public void register( Task task ) throws Exception {

        this.tasks.put( task.getIdentifier(), task );
        
    }

    /**
     * Unregister the given task so that it doesn't process any more.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: TaskManager.java,v 1.15 2002/07/03 22:23:36 burton Exp $
     */
    public void unregister( String identifier ) throws Exception {

        this.tasks.remove( identifier );
        
    }
    
    /**
     * If the Task system is enabled, return true.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public static boolean checkTaskSystemDisabled() {

        boolean restarted = SystemProperties.getBoolean( SystemProperties.REPTILE_SYSTEM_RESTARTED_KEY );

        boolean tasksDisabled = SystemProperties.getBoolean( SystemProperties.REPTILE_SYSTEM_TASKS_DISABLED_KEY );
        
        if ( restarted || tasksDisabled ) {

            Log.warn( "Reptile tasks restarted or disabled." );

            return true;
        } 

        return false;
        
    }

    /**
     * Get a task from the TaskManager.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public Task getTask( String identifier ) throws Exception {

        return (Task)tasks.get( identifier );
        
    }

    /**
     * Return the total number of tasks.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getTaskCount() {

        return tasks.size();

    }

    /**
     * Return true if ANY tasks are currently running.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public boolean anyTasksRunning() {

        Enumeration tasks = this.getTasks();

        while ( tasks.hasMoreElements() ) {

            Task task = (Task)tasks.nextElement();

            if ( task.isRunning() ) {

                return true;
                
            } 
            
        } 

        return false;
        
    }
    
    /**
     * Destroy the tasks system.  Basically this sets the tasks systems to
     * disabled (so that nothing else starts up again) and then waits until all
     * tasks are not running.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void destroy() {

        System.out.print( "Destroying tasks system..." );
        
        System.setProperty( SystemProperties.REPTILE_SYSTEM_TASKS_DISABLED_KEY, "true" );

        //wait until running tasks have stopped.

        long maxtime = 60000; // wait a maximum of 60 seconds...

        long started = System.currentTimeMillis();
        
        while ( this.anyTasksRunning() ) {

            try { 
                
                Thread.sleep( 300 );
                
            } catch ( Throwable t ) { }

            System.out.print( "." );

            if ( System.currentTimeMillis() - started > maxtime ) {

                System.out.println( "Maxtime for stopping tasks has exceeded: " + maxtime );
                
                break; //don't need to wait any longer
            } 

        } 

        System.out.println( "" );
        
        System.out.println( "Destroying tasks system...done" );
        
    }

    /**
     * Run a specific task, catching all exceptions and logging them correctly.
     *
     * <p>
     * This is the best way to run a task.  Do not call run() directly.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public static void runTask( Task task ) {

        try {

            //figure out if we should run the task..

            if ( task.getLastRan() + task.getInterval() < System.currentTimeMillis() ) {

                //run the task.  When first called all tasks will run
                //because getLastRan will return 0.

                task.setState( Task.STATE_RUNNING );

                Log.message( "TaskThread: running task - " + task.getIdentifier(), task );
                    
                task.run();
                    
                task.setState( Task.STATE_PAUSED );
                    
                task.setLastRan( System.currentTimeMillis() );

                task.setLastMessage( "" );

                TaskLogPeer.success( task );

            } 

        } catch ( Throwable t ) {

            task.setState( Task.STATE_FAILED );
                    
            task.setLastMessage( t );

            task.setLastRan( System.currentTimeMillis() );
            
            TaskLogPeer.failed( task, t );
                    
            Log.error( task, t );
                    
        }

    }
    
}
