/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.services;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.tasks.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;
import org.openprivacy.reptile.search.*;
import org.openprivacy.reptile.search.requests.*;

import org.apache.turbine.om.peer.*;

import org.jxta.bridge.ServiceDescriptor;

/**
 * Provides a search service that 3rd parties (peers, developers) can work with.
 *
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: ReptileSearchService.java,v 1.3 2002/03/17 07:29:07 burton Exp $
 */
public class ReptileSearchService {

    public static final ServiceDescriptor DESCRIPTOR =
        new ServiceDescriptor( ReptileSearchService.class.getName(),
                               "urn:reptile/jxta/service/ReptileSearchService",
                               "1.0.0",
                               "OpenPrivacy.org",
                               "http://reptile.openprivacy.org",
                               "A service this allow you to say hello.", 
                               System.getProperty( SystemProperties.REPTILE_JXTA_PEERGROUP_ID_KEY ),
                               System.getProperty( SystemProperties.REPTILE_JXTA_PEERGROUP_NAME_KEY ),
                               System.getProperty( SystemProperties.REPTILE_JXTA_PEERGROUP_DESCRIPTION_KEY ),
                               true );

    public static final String[] ADVANCED_SEARCH_REQUESTS = { "UnreadArticlesSearchRequest",
                                                              "RSSWeblogSearchRequest",
                                                              "RecentArticlesSearchRequest",
                                                              "NewestArticlesSearchRequest",
                                                              "ChannelHistorySearchRequest",
                                                              "BrowseWeblogSearchRequest",
                                                              "BrowseChannelsSearchRequest" };

    public static final String[] SEARCH_PROVIDERS = { "ArticleSearchProvider",
                                                      "ChannelSearchProvider",
                                                      "WeblogSearchProvider" };    

    private Vector requests = new Vector();
    private Vector providers = new Vector();
    
    /**
     * 
     * Create a new <code>ReptileSearchService</code> instance.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public ReptileSearchService() {

        include( ADVANCED_SEARCH_REQUESTS, requests );
        include( SEARCH_PROVIDERS, providers );
        
    }

    /**
     * Given a name of an AdvancedSearchRequest, execute it.  We return a handle
     * which can be used to navigate requests in the future.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String searchByAdvancedSearchRequest( String name ) throws Exception {

        assertValidRequest( name );

        AdvancedSearchRequest req =(AdvancedSearchRequest)Class
            .forName( SearchProviderManager.ADVANCED_SEARCH_REQUEST_PACKAGE + name ).newInstance();

        Class cprovider = req.getRequiredSearchProvider();

        if ( cprovider == null ) {

            throw new Exception( "No provider specified." );
            
        } 

        SearchProvider provider = SearchProviderManager.getInstance()
            .newProvider( cprovider );

        provider.search( req );
        
        return provider.getHandle();
        
    }

    /**
     * Given a name of an AdvancedSearchRequest, execute it.  We return a handle
     * which can be used to navigate requests in the future.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String search( String criteria, String provider ) throws Exception {

        assertValidProvider( provider );

        Class cprovider = Class.forName( SearchProviderManager.SEARCH_PROVIDER_PACKAGE + provider );

        SearchProvider p = SearchProviderManager.getInstance()
            .newProvider( cprovider );

        SearchRequest req = new SearchRequest();
        req.getCriteria().add( criteria );
        
        p.search( req );
        
        return p.getHandle();
        
    }

    /**
     * Get the advanced search request that are supported.
     */
    public String[] getAdvancedSearchRequests() {

        return ADVANCED_SEARCH_REQUESTS;
        
    }

    /**
     * Return the list of providers this service supports
     */
    public String[] getSearchProviders() {

        return SEARCH_PROVIDERS;
        
    }

    /**
     * Import the given object array into the vector.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void include( Object[] objs, Vector v  ) {

        for ( int i = 0; i < objs.length; ++i ) {

            v.addElement( objs[i] );
            
        } 

    }

    /**
     * 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void assertValidRequest( String request ) throws Exception {

        if ( requests.contains( request ) == false ) {
            throw new Exception( "Invalid request: " + request );
        } 

    }

    /**
     * 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private void  assertValidProvider( String provider ) throws Exception {

        if ( providers.contains( provider ) == false ) {
            throw new Exception( "Invalid provider: " + provider );
        } 

    }

    /**
     * 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String serialize( String handle, int page ) throws Exception {

        SearchProvider provider = SearchProviderManager.getInstance().getSearchProvider( handle );

        SearchSerializer serializer = new SearchSerializer( provider );

        return serializer.serializeAsString( page );
        
    }

    /**
     * Get the give records and return them as an array of SearchRecords.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchRecord[] records( String handle, int begin, int end ) throws Exception {

        Vector v = new Vector();

        SearchProvider provider = SearchProviderManager.getInstance().getSearchProvider( handle );

        for ( int i = begin; i <= end; ++i  ) {

            SearchRecord record = new SearchRecord();
            record.setIndex( i );
            provider.record( record );

            v.addElement( record );

        } 

        SearchRecord[] records = new SearchRecord[ v.size() ];
        v.copyInto( records );

        return records;
        
    }

    /**
     * Get the result count for this.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getResultCount( String handle ) throws Exception {

        SearchProvider provider = SearchProviderManager.getInstance().getSearchProvider( handle );
        
        return provider.getResultCount();
        
    }

}
