/*
 * ---- 
 *
 * $Id: DBSearchProvider.java,v 1.8 2002/07/06 05:57:57 burton Exp $
 * $Project: http://reptile.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/sierra/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ----
 *
 * Copyright 2001 OpenPrivacy.org  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package org.openprivacy.reptile.search.impl;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.om.*;
import org.openprivacy.reptile.search.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

// Village classes
import com.workingdogs.village.*;

// Turbine classes
import org.apache.turbine.om.peer.*;
import org.apache.turbine.util.*;
import org.apache.turbine.util.db.*;
import org.apache.turbine.util.db.map.*;
import org.apache.turbine.util.db.pool.DBConnection;

// Local classes
import org.openprivacy.reptile.om.map.*;

import java.lang.reflect.*;

/**
 * A generic SearchProvider for the local DB.  This should contain the default
 * fields and sort order and the Class of the peer to use.
 * 
 * @see SearchProvider
 * @see AbstractSearchProvider
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: DBSearchProvider.java,v 1.8 2002/07/06 05:57:57 burton Exp $
 */
public abstract class DBSearchProvider extends AbstractSearchProvider {

    private Class peer = null;

    /**
     * Stores query results
     */
    private Vector results = null;

    /**
     * Default fields for this search provider.
     */
    protected List defaultSearchFields = new ArrayList();

    /**
     * Default sort order for this search provider.
     */
    protected List defaultSortOrder = new ArrayList();

    /**
     * 
     * Create a new <code>DBSearchProvider</code> instance.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public DBSearchProvider( Class peer ) {

        this.peer = peer;
        
    }

    /**
     * 
     * @see SearchProvider#search
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void search( SearchRequest sr ) throws Exception {

        //make sure we can search...
        this.assertSearchable();

        //make sure that this request is valid.
        sr.assertValid();

        this.setSearchRequest( sr );
        
        this.setState( SearchProvider.STATE_SEARCH_IN_PROGRESS );

        //build the search and execute it.

        //make sure we have at least one field

        if ( sr.getSearchFields().size() == 0 ) {

            Iterator dfi = defaultSearchFields.iterator();

            while ( dfi.hasNext() ) {

                sr.getSearchFields().add( dfi.next() );
                
            } 

        }
        
        //make sure we have a sort order...
        if ( sr.getSortOrder().size() == 0 ) {

            Iterator sfi = defaultSortOrder.iterator();

            while ( sfi.hasNext() ) {

                sr.getSortOrder().add( sfi.next() );
                
            } 

        }

        Iterator fields = sr.getSearchFields().iterator();
        
        Iterator order = sr.getSortOrder().iterator();

        String field;

        //criteria as a string.
        Object search_string;

        String comparator = getSearchRequest().getComparator();

        Criteria criteria = new Criteria();

        Criteria.Criterion criterion = null;
        Criteria.Criterion last_crit = null;

        criteria.setIgnoreCase( true );

        String tablename = this.getTableName();
        
        while ( fields.hasNext() ) {

            field = (String)fields.next();

            //Due to a bug (feature?) in turbine we have to make sure the field
            //begins with the table name.  IE TABLE.COLUMN

            if ( field.startsWith( tablename ) == false ) {
                field = tablename + "." + field.toUpperCase();
            } 

            Iterator search_criteria = sr.getCriteria().iterator();
            
            while ( search_criteria.hasNext() ) {

                search_string = search_criteria.next();
                
                //always match the whole string.
                if ( comparator.equals( SearchRequest.LIKE ) && search_string instanceof String ) {
                    search_string = "%" + (String)search_string + "%";
                } 

                Criteria.Criterion c = criteria.getNewCriterion( field, search_string, comparator );

                if ( criterion == null ) {

                    criterion = c;
                    
                } else {

                    last_crit.or( c );
                    
                }
                
                last_crit = c;
                
            } 
            
        }

        criteria.add( criterion );
        
        //set the search order.
        while ( order.hasNext() ) {

            criteria.addDescendingOrderByColumn( (String)order.next() );
            
        }

        //FIXME: use reflection here...

        Class[] params = new Class[] { Criteria.class };
        
        Method doSelect = peer.getMethod( "doSelect", params );

        Object[] args = new Object[] { criteria };

        //set the max rows for this request
        if ( this.requiresMaxRows() ) {

            criteria.setLimit( sr.getMaxCount() );
            
        }
        
        this.results = (Vector)doSelect.invoke( null, args );
        
        this.setResultCount( results.size() );
        
        this.setState( SearchProvider.STATE_SEARCH_COMPLETE );
        
    }

    /**
     * Return true if we need to set MAXROWS
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private boolean requiresMaxRows() {

        SearchRequest sr = this.getSearchRequest();
        
        return sr.getMaxCount() != SearchRequest.NO_MAXCOUNT && sr.getMaxCount() > 0;
        
    }
    
    /**
     * Return the results directly from the search provider for use by class
     * implementations for pulling out records.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    protected Vector getResults() {

        return results;
        
    }

    /**
     * Get the table name from the BasePeer class that this 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private String getTableName() throws Exception {

        // Get the TABLE_NAME member from the peer instance

        return (String)peer.getField( "TABLE_NAME" ).get( null );
        
    }

}
