/*
 * ---- 
 *
 * $Id: SearchSerializer.java,v 1.19 2002/02/19 01:33:46 burton Exp $
 * $Project: http://reptile.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/sierra/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ----
 *
 * Copyright 2001 OpenPrivacy.org  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package org.openprivacy.reptile.search;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.om.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

import talon.*;
import talon.components.*;
import talon.resources.*;
import talon.util.*;
import talon.util.net.*;

import org.jdom.*;
import org.jdom.output.*;

/**
 * Provides a mechanism for searializing pages of a SearchProviders results into
 * an XML representation.
 * 
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: SearchSerializer.java,v 1.19 2002/02/19 01:33:46 burton Exp $
 */
public class SearchSerializer {
    
    public static final int FIRST_PAGE = 0;

    public static final int RECORDS_PER_PAGE = 10;

    private SearchProvider provider = null;
    
    /**
     * 
     * Create a new <code>SearchSerializer</code> instance.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchSerializer( SearchProvider provider ) {

        this.setProvider( provider );
        
    }

    /**
     * Determine the number of 'pages' that this SearchProvider found in the last
     * search.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getPageCount() {

        return 1 + (int)Math.floor( ( provider.getResultCount() - 1 ) / RECORDS_PER_PAGE );
        
    }

    /**
     * <p> Serialize the given page a XML.  Note that if you specify an invalid
     * page we will throw an exception.
     *
     * <p> All page indexes begin at 0.
     * 
     * <p> The XML format we should return is the standard Reptile search
     * markup.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public Element serialize( int page ) throws Exception {

        //validate the page input
        if ( page > getPageCount() || page < 0 )
            throw new Exception( "Invalid page: " + page );
        
        //get the start index
        
        int start = page * RECORDS_PER_PAGE;
        int end = start + RECORDS_PER_PAGE - 1; //offset by one because we use 0 based indexes.

        SearchRecord visitor = new SearchRecord();

        Element search = new Element( "search", ReptileNamespaces.JDOM_SEARCH );

        //set the provider-handle
        search.setAttribute( "provider-handle", provider.getHandle() );

        int state = provider.getState();
        
        //set the provider-handle
        search.setAttribute( "provider-state", provider.getState( state ) );

        search.setAttribute( "provider-name", getClassWithoutPackage( provider ) );
        search.setAttribute( "request-name", getClassWithoutPackage( provider.getSearchRequest() ) );

        //update search time info
        
        long started = provider.getSearchStartTime();
        long completed = provider.getSearchCompletedTime();
            
        search.setAttribute( "search-started", Long.toString( started ) );
        search.setAttribute( "search-completed", Long.toString( completed ) );

        search.setAttribute( "search-time", Long.toString( completed - started ) );
        
        SearchRequest request = provider.getSearchRequest();

        if ( request != null ) {
            search.addContent( request.serialize() );
        } else {
            Log.warn( "Provider has not set its SearchRequest: " + provider.getClass().getName() ) ;
        }

        Element results = new Element( "results", ReptileNamespaces.JDOM_SEARCH );

        int index;

        //the total number of items I am going to return
        int returning = 0;
        
        for ( index = start; index <= end && index < provider.getResultCount(); ++index ) {

            //set the index so that we know which entry to pull out.
            visitor.setIndex( index );
            
            provider.record( visitor );

            results.addContent( visitor.serialize() );

            ++returning;
            
        } 

        if ( returning == RECORDS_PER_PAGE ) { --index; } 
        
        
        results.setAttribute( "start", Long.toString( start ) );

        //set the end to the last index value.

        results.setAttribute( "end", Long.toString( index ) );

        results.setAttribute( "found", Long.toString( provider.getResultCount() ) );
        results.setAttribute( "total", Long.toString( ArticlePeer.getCount() ) );
        results.setAttribute( "page", Long.toString( page ) );
        results.setAttribute( "total-pages", Long.toString( this.getPageCount() ) );

        search.addContent( results );

        return search;

    }

    /**
     * Serialize this page as a String (probably for use within a test or for
     * human evaluation.)
     * 
     * @see #serialize
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String serializeAsString( int page ) throws Exception {

        XMLOutputter outputter = new XMLOutputter();

        outputter.setEncoding( ReptileResources.DEFAULT_XML_ENCODING );
        
        return outputter.outputString( serialize( page ) );

    }

    /**
     * Serialize this page as a W3C DOM Node (probably for use within Xalan)
     * 
     * @see #serialize
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public org.w3c.dom.Node serializeAsNode( int page ) throws Exception {

        return new DOMOutputter().output( serialize( page ) );

    }

    public org.w3c.dom.Document serializeAsDocument( int page ) throws Exception {

        return new DOMOutputter().output( new Document( serialize( page ) ) );

    }

    /**
     * 
     * Get the value of <code>provider</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider getProvider() { 
        
        return this.provider;
        
    }

    /**
     * 
     * Set the value of <code>provider</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setProvider( SearchProvider provider ) { 
        
        this.provider = provider;
        
    }

    /**
     * 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    private String getClassWithoutPackage( Object object ) {

        if ( object == null ) {

            return "not-available";
            
        } else {

            String name = object.getClass().getName();

            name = name.substring( name.lastIndexOf( "." ) + 1, name.length() );

            return name;

        }
        
    }
    
}
