/*
 * ---- 
 *
 * $Id: SearchRequest.java,v 1.12 2002/02/24 22:45:07 burton Exp $
 * $Project: http://reptile.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/sierra/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ----
 *
 * Copyright 2001 OpenPrivacy.org  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package org.openprivacy.reptile.search;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

import org.apache.commons.collections.*;

import org.jdom.*;
import org.jdom.output.*;

/**
 * <p> A SearchRequest is used to represent a new search we want to run on a
 * SearchProvider.  All search query abstraction is handled here.
 *
 * <p> All options need to be specified here before a search can be executed.
 *
 * <p> In order to execute a search provider on you just need to specify the
 * fields you should search on.
 * 
 * <code>
 * getFields().add( "title" );
 * </code
 * 
 * <p> ... and then specify a string or strings to search for as the criteria.
 *
 * <p> You can then specify the boolean operator for this request with the
 * <b>setBooleanOperator</b> method.  
 *
 * <p> In some situations this may not be enough information necessary for a
 * SearchProvider to excute a search.  For example if you need to restrict your
 * information to a certain peergroup or some other functionaity which is
 * specific to a provider.
 *
 * <p> There are two solutions for this:
 * 
 * <ol>
 * 
 * <li> Implement a custom search request byte extending SearchRequest.  You can
 * then place any necessary additional information int your new class.
 * 
 * <li> Use the Criteria object to restrict items on a per field basis.
 * 
 * </ol>
 * 
 * <p> Either strategy is acceptable, you should just make the most appropriate
 * decision for your situation.
 * 
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: SearchRequest.java,v 1.12 2002/02/24 22:45:07 burton Exp $
 */
public class SearchRequest {

    /**
     * Boolean OR operator
     */
    public static final int BOOLEAN_OR = 1;

    /**
     * Boolean AND operator
     */
    public static final int BOOLEAN_AND = 2;

    /**
     * Comparison type.
     */
    public static final String EQUAL = "=";

    /**
     * Comparison type.
     */
    public static final String NOT_EQUAL = "<>";

    /**
     * Comparison type.
     */
    public static final String ALT_NOT_EQUAL = "!=";

    /**
     * Comparison type.
     */
    public static final String GREATER_THAN = ">";

    /**
     * Comparison type.
     */
    public static final String LESS_THAN = "<";

    /**
     * Comparison type.
     */
    public static final String GREATER_EQUAL = ">=";

    /**
     * Comparison type.
     */
    public static final String LESS_EQUAL = "<=";

    /**
     * Comparison type.
     */
    public static final String LIKE = " LIKE ";

    // public static final String STRING_EQUAL = "='";

    /**
     * Comparison type.
     */
    public static final String CUSTOM = "CUSTOM";

    // public static final String FOREIGN_KEY = "fk";

    /**
     * Comparison type.
     */
    public static final String DISTINCT = "DISTINCT ";

    /**
     * Comparison type.
     */
    public static final String IN = " IN ";

    /**
     * Comparison type.
     */
    public static final String NOT_IN = " NOT IN ";

    /**
     * Comparison type.
     */
    public static final String ALL = "ALL ";

    /**
     * Comparison type.
     */
    public static final String JOIN = "JOIN";

    /**
     * "Order by" qualifier - ascending
     */
    private static final String ASC = "ASC";

    /**
     * "Order by" qualifier - descending
     */
    private static final String DESC = "DESC";

    public static final int NO_MAXCOUNT = -1;
    
    private List criteria = new ArrayList();
    private List searchFields = new ArrayList();
    private List sortOrder = new ArrayList();

    /**
     * Maximum number of records that this search should return.
     */
    private int maxCount = NO_MAXCOUNT;
    
    /**
     * Handles extended properties for this SearchRequest.
     */
    private ExtendedProperties ep = new ExtendedProperties();

    private int booleanOperator = BOOLEAN_OR;

    private String comparator = LIKE;

    /**
     * <p> This represents the data you are searching for.
     *
     * <p> For example if you want to search for 'linux' you would set the query
     * to 'linux'.
     * 
     * <p> In a lot of frameworks this is called a 'term'.  I wanted to avoid
     * use of that definition because it is slightly confusing in an API.
     *
     * <p> The way your criteria are handled is up to the SearchProvider.
     * Generally it is a good idea to tokenize before hand but it is possible to
     * let your SearchProvider to handle tokenizing for you.
     * 
     * <p> This is required.  By default it is 'null'
     *
     * <p> IMPLEMENTATION NOTE: Right now this only supports Strings.  in the
     * future we might create a ComplexSearchCriteria object so that you can
     * specify more complex queries.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public List getCriteria() { 
        
        return this.criteria;
        
    }

    /**
     * Add a criteria to this request.
     *
     * <p>
     * This is required.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void addCriteria( String criteria ) { 
        
        this.criteria.add( criteria );
        
    }

    /**
     * Add a criteria to this request.
     *
     * <p>
     * This is required.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void addCriteria( Long criteria ) { 
        
        this.criteria.add( criteria );
        
    }

    /**
     * @see #getCriteria
     * @see #addCriteria
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setCriteria( List criteria ) {

        this.criteria = criteria;
        
    }

    /**
     * Get the fields that this search applies to.  For example in a database
     * query you could specify the fields 'title' and 'description'.  
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public List getSearchFields() {

        return this.searchFields;
        
    }

    /**
     * Add a search field.
     *
     * @see #getSearchFields
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void addSearchField( String field ) {

        this.searchFields.add( field );
        
    }

    /**
     * @see #getSearchFields
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setSearchFields( List searchFields ) {

        this.searchFields = searchFields;

    }

    /**
     *
     * <p>
     * Get the sort order for this SearchRequest.
     * 
     * <p>
     * Basically this is a list of fields that we need to sort by.
     * 
     * <p>
     * Note that all SearchProviders do not need to support this.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public List getSortOrder() {

        return this.sortOrder;
        
    }

    /**
     *
     * @see #getSortOrder
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void addSortOrder( String field ) {

        this.sortOrder.add( field );
        
    }

    /**
     * @see #getSortOrder
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setSortOrder( List sortOrder ) {

        this.sortOrder = sortOrder;

    }

    /**
     * 
     * Get the value of <code>maxCount</code>.
     *
     * <p>
     * This is optional, by default we use NO_MAXCOUNT
     *
     * <p>
     * Note that all SearchProviders may not implement this.  For example at the
     * time of this writing Lucene does not support this.
     *     
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getMaxCount() { 
        
        return this.maxCount;
        
    }

    /**
     * Set the value of <code>maxCount</code>.
     *
     * @see #getMaxCount
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setMaxCount( int maxCount ) { 
        
        this.maxCount = maxCount;
        
    }

    /**
     * <p>
     * Allows you to specify ExtendedProperties for this SearchRequest.  A
     * SearchProvider implementation can support any type of named search
     * properties what can be set here.
     * 
     * <p>
     * It is up to the SearchProvider implementation to document these
     * correctly.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public ExtendedProperties getExtendedProperties() {

        return this.ep;

    }

    /**
     * Serialize this as XML suitable for using as an XML results set.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public Element serialize() {

        Element request = new Element( "request", ReptileNamespaces.JDOM_SEARCH );

        request.setAttribute( "max-count", Integer.toString( getMaxCount() ) );
        
        request.addContent( serializeList( "criteria", getCriteria() ) );
        request.addContent( serializeList( "search-fields", getSearchFields() ) );
        request.addContent( serializeList( "sort-order", getSortOrder() ) );

        return request;
        
    }
    
    private Element serializeList( String parent, List list ) {
        
        Element result = new Element( parent, ReptileNamespaces.JDOM_SEARCH );
        
        Iterator it = list.iterator();

        while ( it.hasNext() ) {

            String next = it.next().toString();

            //FIXME: in the future actually specify the type if this isn't a string (AKA a long)
            result.addContent( new Element( "string", ReptileNamespaces.JDOM_XS ).setText( next ) );
            
        } 

        return result;
        
    }
    
    /**
     * Assert that this request is valid, that it specifies a criteria, etc.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void assertValid() throws Exception {

        if ( this.getCriteria().size() == 0 )
            throw new Exception( "Need to specify criteria." );
        
    }

    /**
     * 
     * Get the default type of boolean operation this should support.  This is
     * only really necessary if you have lists of strings as criteria.
     *
     * @see #BOOLEAN_OR
     * @see #BOOLEAN_AND
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getBooleanOperator() { 
        
        return this.booleanOperator;
        
    }

    /**
     * 
     * Set the value of <code>booleanOperator</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setBooleanOperator( int booleanOperator ) { 
        
        this.booleanOperator = booleanOperator;
        
    }

    /**
     * 
     * Get the value of <code>comparator</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String getComparator() { 
        
        return this.comparator;
        
    }

    /**
     * 
     * Set the value of <code>comparator</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setComparator( String comparator ) { 
        
        this.comparator = comparator;
        
    }

}
