/*
 * ---- 
 *
 * $Id: SearchProviderManager.java,v 1.9 2002/03/17 06:44:11 burton Exp $
 * $Project: http://reptile.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/sierra/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ----
 *
 * Copyright 2001 OpenPrivacy.org  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package org.openprivacy.reptile.search;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

import talon.*;
import talon.components.*;
import talon.resources.*;
import talon.util.*;
import talon.util.net.*;

/**
 * 
 * Handles creating and managing SearchProviders.  
 * 
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: SearchProviderManager.java,v 1.9 2002/03/17 06:44:11 burton Exp $
 */
public class SearchProviderManager {

    public static final String SEARCH_PROVIDER_PACKAGE
        = "org.openprivacy.reptile.search.impl.";

    public static final String ADVANCED_SEARCH_REQUEST_PACKAGE
        = "org.openprivacy.reptile.search.requests.";

    /**
     * Instance member for <code>SearchProviderManager</code>
     */
    private static SearchProviderManager instance = null;

    private HashMap providers = new HashMap();
    
    /**
     * 
     * Get an instance of the <code>SearchProviderManager</code>.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public static SearchProviderManager getInstance() {
        
        if ( instance == null ) {
            
            instance = new SearchProviderManager();
            
        }
        
        return instance;
        
    }

    /**
     * Create a new SearchProvider, throw an Exception if there was a problem.
     *
     * We also instantiate the request name and run a search with it.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider newProvider( String classname,
                                       String request_name ) throws Exception {

        SearchProvider provider = newProvider( Class.forName( classname ) );

        SearchRequest request = (SearchRequest)Class.forName( request_name ).newInstance();

        provider.search( request );

        return provider;
        
    }
    
    /**
     * Create a new SearchProvider, throw an Exception if there was a problem.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider newProvider( String classname ) throws Exception {

        return newProvider( Class.forName( classname ) );

    }

    /**
     * Create a new SearchProvider, throw an Exception if there was a problem.
     *
     * @param expand if true we will expand the short version of this classname
     * 'ArticleSearchProvider' into a full classname
     * 'org.openprivacy.reptile.search.impl.ArticleSearchProvider'
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider newProvider( String classname, boolean expand ) throws Exception {

        if ( expand )
            classname = SEARCH_PROVIDER_PACKAGE + classname;

        return newProvider( Class.forName( classname ) );

    }

    /**
     * Create a new SearchProvider, throw an Exception if there was a problem.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider newProvider( Class cprovider ) throws Exception {

        SearchProvider sp = (SearchProvider)cprovider.newInstance();
        
        //any initialization

        //put this provider in the list of providers.
        providers.put( sp.getHandle(), sp );
        
        return sp;

    }
    
    /**
     * Destroy the given SearchProvider and remove all evidence that it ever
     * existed.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void destroy( SearchProvider provider ) {

        Log.debug( "Destroying SearchProvider: " + provider.getHandle() );
        
        try {

            provider.destroy();

            this.providers.remove( provider );

            provider = null;
            
        } catch ( Throwable t ) {

            Log.error( t );
            
        }

        //remove the given provider so that it isn't available anymore.
        providers.remove( provider.getHandle() );

        provider = null;
        
    }
    
    /**
     * Get a collection of all instantiated SearchProviders.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public Iterator getSearchProviders() {

        return providers.values().iterator();
        
    }

    /**
     * Get the total number of instantiated SearchProviders.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getSearchProviderCount() {

        return providers.values().size();
        
    }

    /**
     * Get an instantiated search provider (which is probably running or has ran
     * a search) by its handle.  This should be a provider that was already
     * instantiated by newProvider.
     *
     * @see #newProvider
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchProvider getSearchProvider( String handle ) throws Exception {

        //FIXME: there could be a security issue here.  We don't want to provide
        //a SearchProvider which many contain controversial subject to another
        //remote user.
        
        SearchProvider sp = (SearchProvider)this.providers.get( handle );

        if ( sp == null )
            throw new Exception( "Invalid search provider handle: " + handle );
        
        return sp;
        
    }
    
}
