/*
 * ---- 
 *
 * $Id: SearchProvider.java,v 1.14 2002/02/07 11:08:13 burton Exp $
 * $Project: http://reptile.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/sierra/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 *
 * ----
 *
 * Copyright 2001 OpenPrivacy.org  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have obtaind with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 */

package org.openprivacy.reptile.search;

import java.io.*;
import java.net.*;
import java.util.*;

import org.openprivacy.reptile.actions.*;
import org.openprivacy.reptile.extensions.*;
import org.openprivacy.reptile.init.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

/**
 * <p>
 * The SearchProvider class provides a plugin mechanism for integrating external
 * search mechanisms within Reptile.
 * 
 * Essentially the goals are:
 * 
 * <p>
 *
 * <li>Support asynchronous code such as JXTA, Freenet.
 * 
 * <li>Support synchronous code like local DB queries via JDBC.
 * 
 * <li>Support pageable result sets int a portable manner.
 * 
 * <li>Allow us to support future implementations of SearchProviders.
 * 
 *
 * <p>
 * 
 * The design of the SearchProvider was based on the following search frameworks.
 * 
 * <p>
 * 
 * <li>Conventional DB searching.  ( SELECT statements )
 * 
 * <li>The Lucene search framework ( http://jakarta.apache.org/lucene )
 * 
 * <li>The JXTA search framework ( http://search.jxta.org )
 * 
 * <p>
 * Note: if you run a search on a SearchProvider you are NOT guaranteed that it
 * will return a result immediately.  In fact it might need to run for a while
 * so that we can collect results.
 * 
 * <p>
 * If you depend on a stable result set, you should synchronize around this
 * search provider.  This is important especially when you want to serialize a
 * group of results
 * 
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: SearchProvider.java,v 1.14 2002/02/07 11:08:13 burton Exp $
 */
public interface SearchProvider {

    /**
     * The state of the SearchProvider if there has NEVER been a search run on
     * it.
     */
    public static final int STATE_NEVER_SEARCHED          = 1;

    /**
     * The state of this SearchProvider if it is still trying to find results.
     */
    public static final int STATE_SEARCH_IN_PROGRESS      = 2;

    /**
     * Should be the state of this SearchProvider if the given search is
     * complete.
     */
    public static final int STATE_SEARCH_COMPLETE         = 4;
    
    /**
     *
     * <p>
     *
     * Get the number of results found for this SearchProvider.  If the search
     * has not been run yet this will return 0.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getResultCount();

    /**
     *
     * <p>
     *
     * Set the result count.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setResultCount( int resultCount );

    /**
     *
     * <p>
     *
     * Get a unique handle for this SearchProvider.  Each SearchProvider has a
     * unique handle that can be used to obtain it from the
     * SearchProviderManager
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String getHandle();

    /**
     *
     * <p>
     *
     * The time created for this this SearchProvider is the current time in
     * milliseconds when this SearchProvider was instantiated.
     *
     * <p>
     *
     * This is used by the SearchProviderGCThread to determine when it should GC
     * SearchProviders.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public long getTimeCreated();    

    /**
     * <p>
     *
     * Run a search with a given query.  Note that some search providers may NOT
     * finish the search here..  Specifically some search providers may take a
     * few minutes to fill in results as results are collected.  A good example
     * of this is code that executes within a P2P network and is waiting or
     * other peers to reply.
     *
     * <p>
     *
     * Only one 'search' can be run on any search provder.  If you wish to run
     * multiple searches you need to instantiate multiple SearchProviders.
     * 
     * <p>
     *
     * If you want to develop a mutltithreaded/asynchronous SearchProvider, you
     * should set the state to STATE_SEARCH_IN_PROGRESS, kick off any threads or
     * async queries an then set the state to STATE_SEARCH_COMPLETE (possibly
     * outside of the search method)
     * 
     * <p>
     *
     * Note that when you are running this method you should synchronize around
     * 'this' object instance.  This should only be done int code that could
     * break the search if it is requested to early.
     * 
     * <p>
     *
     * AKA This IS thread reentrant!
     * 
     * <p>
     *
     * SearchProviders also need to pay attention to the SearchRequest and
     * implement any settings here.  Specifically we need to pay attention to
     * sort order, search fields, criteria, etc.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void search( SearchRequest sr ) throws Exception;

    /**
     * <p>
     *
     * Get the given SearchRecord from this SearchProvider.
     *
     * <p>
     *
     * It is important that all SearchProviders implement this method correctly.
     *
     * <p>
     *
     * The given implementation needs to fill in all necessary information into
     * this SearchRecord, this includes the title, description, etc.
     *
     * <p>
     *
     * Note.  This is basically a visitor pattern.  Callers use a search
     * provider and then fetch a record from the SearchProvider by index.
     *
     * @throws Exception When an invalid SearchRecord index is used
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void record( SearchRecord record ) throws Exception;

    /**
     * <p>
     *
     * Called before garbage collection by the SearchProviderGCThread.  If your
     * search provider needs to perform any garbage collection it should do so
     * here.
     *
     * <p>
     *
     * If your SearchProvider has executed an asynchronous query, it should
     * abort this before returning.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void destroy() throws Exception;

    /**
     * Get the state of this SearchProvider
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public int getState();

    /**
     * Get the state of this provider as a string.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public String getState( int state );
    
    /**
     * Set the state of this SearchProvider
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setState( int state );

    /**
     * Return true if this SearchProvider is searchable.  A SearchProvider is
     * only searchable if it's state is STATE_NEVER_SEARCHED
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public boolean isSearchable();

    /**
     * Throws an Exception if this SearchProvider is not searchable.
     * 
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void assertSearchable() throws Exception;

    /**
     * Set the SearchRequest that this provider has results for.  This should be
     * called at least once when a SearchProvider has ran a search.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public SearchRequest getSearchRequest();
    
    /**
     * 
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setSearchRequest( SearchRequest request );

    /**
     * Get the time (UNIX time) that this providers search was started on.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public long getSearchStartTime();

    /**
     * 
     * @see #getSearchStartTime
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setSearchStartTime( long time );
    
    /**
     * Get the time (UNIX time) that this providers search was completed on.
     *
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public long getSearchCompletedTime();

    /**
     * 
     * @see #getSearchCompletedTime
     * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
     */
    public void setSearchCompletedTime( long time );

}
