/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.extensions;

import java.io.*;
import java.net.*;
import java.util.*;
import java.text.*;

import org.apache.xalan.extensions.*;
import org.apache.xalan.templates.*;

import org.openprivacy.panther.components.*;

import org.openprivacy.reptile.util.*;

import talon.util.*;

/**
 * Misc util functions for obtaining information from the panther proxy.
 * 
 * @author <a href="mailto:burton@openprivacy.org">Kevin A. Burton</a>
 * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
 */
public class ProxyFunctions {

    private Proxy proxy = null;
    
    public ProxyFunctions() {

        this.proxy = ReptileResources.getInstance().getProxy();
        
    }

    /**
     * @see Proxy#contains
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public boolean contains( String location ) {

        return proxy.contains( location );
        
    }

    /**
     * @see Proxy#supports
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $

     */
    public boolean supports( String location ) {

        return proxy.supports( location );
        
    }

    /**
     * @see Proxy#update
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public void update( String location  ) {

        try {

            proxy.update( location );             

        } catch ( Exception e ) {

            Log.error( e );            
        } 
        
    }

    /**
     * Return the fileLocation in the cache that the given URL is found or "" if
     * it isn't in the cache.
     *
     * @see ProxyEntry#getFileLocation
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String getFileLocation( String location ) {

        if ( proxy.contains( location ) ) {

            try {
                 
                return proxy.get( location ).getFileLocation();

            } catch ( IOException e ) {

                Log.error( e );
                
                return "";

            } 
            
        } else {

            return "";

        }
        
    }
    
    /**
     * Get the URL of a specified resource.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String getResourceLocation( String resource ) {

        try { 
            
            return proxy.getResource( resource ).getURL().toString();
            
        } catch ( Throwable t ) {
            
            Log.error( t );
            
            return "";
            
        }

    }

    /**
     * Get the location of a file via it's location.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String file( XSLProcessorContext context,
                        ElemExtensionCall elem ) {
        
        try { 
            
            String location = elem.getAttribute( "location",
                                                 context.getContextNode(),
                                                 context.getTransformer() );

            return getFileLocation( location );
            
        } catch ( Throwable t ) {
            
            Log.error( t );
            
            return "";
            
        }

    }

    /**
     * 
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String lastModified( XSLProcessorContext context,
                                ElemExtensionCall elem ) {

        try {
        
            // end of try-catch
            
            //allow XPATH... 
            String location = elem.getAttribute( "location",
                                                 context.getContextNode(),
                                                 context.getTransformer() );

            if ( proxy.contains( location ) ) {
                
                ProxyEntry entry = proxy.get( location );

                //convert this to seconds...
                long lastmodified = ( System.currentTimeMillis() - entry.getLastModified() );

                return getDateFormated( lastmodified ) + " ago";

            } else {

                //else it is not in the cache... tell the user.

                return "Not currently available";
                
            }
            
        } catch ( Exception e ) {

            Log.error( e );
            
            return "unknown: " + e.getMessage();
            
        }
        
    }

    /**
     * Return a string describing when the next update will happen.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String nextUpdate( XSLProcessorContext context,
                              ElemExtensionCall elem ) {

        try {

            //allow XPATH... 
            String location = elem.getAttribute( "location",
                                                 context.getContextNode(),
                                                 context.getTransformer() );

            if ( proxy.contains( location ) ) {
                
                ProxyEntry entry = proxy.get( location );

                //convert this to seconds...
                long interval = entry.getMinCachedInterval();

                //compute the offset from when this was LAST updated...

                long offset = System.currentTimeMillis() - entry.getLastModified();

                long difference = interval - offset;
                
                return getDateFormated( difference ) + " from now";

            } else {

                //else it is not in the cache... tell the user.

                return "Not currently available";
                
            }

        } catch ( Exception e ) {

            Log.error( e );
            
            return "unknown: " + e.getMessage();
            
        }

    }

    /**
     * Return a string describing when the next update will happen.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public String updateInterval( XSLProcessorContext context,
                                  ElemExtensionCall elem ) {

        try {

            //allow XPATH... 
            String location = elem.getAttribute( "location",
                                                 context.getContextNode(),
                                                 context.getTransformer() );

            if ( proxy.contains( location ) ) {
                
                ProxyEntry entry = proxy.get( location );

                //convert this to seconds...
                long interval = entry.getMinCachedInterval();
                
                return getDateFormated( interval );

            } else {

                //else it is not in the cache... tell the user.

                return "Not currently available";
                
            }

        } catch ( Exception e ) {

            Log.error( e );
            
            return "unknown: " + e.getMessage();
            
        }

    }

    /**
     * Get a formated string representing the difference in time.  Difference
     * should be in seconds.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: ProxyFunctions.java,v 1.11 2002/02/19 01:33:46 burton Exp $
     */
    public static String getDateFormated( long difference ) {

        //convert to seconds

        if ( difference < 1000 ) {

            return "" + difference + " millisecond(s)";
            
        } 

        difference = difference / 1000;
        
        int hours = (int)Math.floor( difference / 3600 );

        int minutes = (int)Math.floor( (difference - (hours * 3600)) / 60 );

        int seconds = (int)difference - hours * 3600 - minutes * 60;

        StringBuffer result = new StringBuffer( "" );
            
        if ( hours > 0 ) {

            result.append( hours + " hour(s), " );
                
        } 

        if ( minutes > 0 ) {

            result.append( minutes + " minute(s), " );
                
        } 

        if ( ( hours > 0  || minutes > 0 ) && seconds > 0 ) {

            result.append( " and " );
                
        } 

        result.append( seconds + " second(s) " );
            
        return result.toString();            

    }
    
}
