/*
 *
 * BEGIN HEADER
 *
 * ---- 
 *
 * $ID: PantherProxy.java,v 1.6 2001/06/12 20:58:43 burton Exp $
 * $Project: http://panther.openprivacy.org $
 * $CVSROOT: :pserver:anoncvs@sierra.openprivacy.org:/usr/local/cvs/public $
 * $WebCVS: http://www.openprivacy.org/cgi-bin/cvsweb/cvsweb.cgi/panther/ $
 * $Mailing-List: http://www.openprivacy.org/lists/ $
 * $Bugzilla: http://bugzilla.openprivacy.org/ $
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * ---- 
 *
 * Copyright 2001 OpenPrivacy.org.  All rights reserved.
 *
 * This program is free software; you can redistribute it and/or modify it under
 * the terms of the LICENSE which you should have received with this package. 
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE.
 *
 * END HEADER
 * 
 */

package org.openprivacy.reptile.cache.sequence;

import java.io.*;
import java.net.*;
import java.util.*;

import javax.xml.transform.*;
import javax.xml.transform.stream.*;

import talon.components.*;
import talon.util.*;
import talon.util.net.*;

import org.openprivacy.reptile.*;
import org.openprivacy.reptile.util.*;
import org.openprivacy.reptile.xslt.*;

/**
 * A handle for items within the sequence cache.
 *
 * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
 * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
 */
public class SequenceCache {
    
    private int cacheMisses = 0;
    
    private int cacheHits = 0;
    
    /**
     * Instance member for <code>SequenceCache</code>
     */
    private static SequenceCache instance = null;

    /*
     * Stores the cache content
     */
    private Hashtable cache = new Hashtable();
    
    public SequenceCache() { }

    /**
     * @see #fetch
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public SequenceCacheEntry fetch( SequenceDispatchEntry sde,
                                     ParameterPasser params ) throws Exception {
        
        return fetch( new SequenceCacheHandle( sde, params ) );
        
    }

    /**
     * Fetch a sequence from the memory cache if it is available.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public SequenceCacheEntry fetch( SequenceCacheHandle handle ) throws Exception {

        String sequenceName = handle.getSequenceDispatchEntry().getName();
        
        if ( contains( handle ) ) {

            SequenceCacheEntry entry = (SequenceCacheEntry)cache.get( handle );

            if ( entry.stale() ) {

                flush( handle );
                
                throw new ReptileException( "Cache entry is stale." );

            }

            return entry;
            
        } else {

            throw new ReptileException( "Sequence is not cacheable: " + sequenceName );
            
        }
        
    }

    /**
     * Return true if the SequenceCache contains the given handle.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public boolean contains( SequenceCacheHandle handle ) {

        return cache.containsKey( handle );
        
    }
    
    /**
     * Store the given handle in the cache with the value of this entry.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    private void store( SequenceCacheHandle handle,
                        SequenceCacheEntry entry ) {
        
        this.cache.put( handle, entry );
        
    }

    /**
     * Store content within the cache.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public void store( SequenceCacheHandle handle, String content ) throws Exception {

        store( handle, new SequenceCacheEntry( handle, content ) );
        
    }

    /**
     * Flush all entries from the cache.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public void flush() {

        synchronized( this ) {

            //start with a new cache and let garbage collection handle the rest.
            cache = null; 
            cache = new Hashtable();

            //reset the hits/misses counters

            setCacheHits( 0 );
            
            setCacheMisses( 0 );
            
        }        

    }
    
    /**
     * Flush the given cache hanle.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public void flush( SequenceCacheHandle handle ) {

        cache.remove( handle );

    }
    
    /**
     * Remove from the cache any entry which points has the given location and contentType..
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public void flush( String location ) {

        //it is important that nothing happens in the cache until we flush...
        synchronized( this ) {
        
            Enumeration enum = cache.keys();

            while ( enum.hasMoreElements() ) {

                SequenceCacheHandle handle = (SequenceCacheHandle)enum.nextElement();

                SequenceDispatchEntry sde = handle.getSequenceDispatchEntry();

                if ( sde.getLocation().equals( location ) ) {

                    Log.message( "Removing content from the cache: " + handle.toString() );
                    flush( handle );
                
                } 
            
            }
        } 
        
    }

    /**
     * Get all known entries from the cache.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public Enumeration getEntries() {

        return cache.elements();
        
    }
    
    /**
     * 
     * Get an instance of the SequenceCache.
     *
     * @author <a href="mailto:burton@relativity.yi.org">burtonator</a>
     * @version $Id: SequenceCache.java,v 1.10 2002/02/27 08:50:53 burton Exp $
     */
    public static SequenceCache getInstance() {
        
        if ( instance == null ) {
            
            instance = new SequenceCache();
            
        }
        
        return instance;
        
    }

    /**
     * 
     * Set the value of <code>cacheHits</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public void setCacheHits( int cacheHits ) { 
        
        this.cacheHits = cacheHits;
        
    }

    /**
     * 
     * Get the value of <code>cacheHits</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public int getCacheHits() { 
        
        return this.cacheHits;
        
    }

    /**
     * 
     * Get the value of <code>cacheMisses</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public int getCacheMisses() { 
        
        return this.cacheMisses;
        
    }

    /**
     * 
     * Set the value of <code>cacheMisses</code>.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public void setCacheMisses( int cacheMisses ) { 
        
        this.cacheMisses = cacheMisses;
        
    }

    /**
     * Increase the cache hits by one.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public void incrementCacheHits() {

        setCacheHits( getCacheHits() + 1 );
        
    }

    /**
     * Increase the cache hits by one.
     *
     * @author <a href="mailto:burton@relativity.yi.org">Kevin A. Burton</a>
     */
    public void incrementCacheMisses() {

        setCacheMisses( getCacheMisses() + 1 );

    }

}
